/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2020 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

#include <cdi.h>

#include <cassert>
#include <algorithm>

#include "percentiles.h"
#include "array.h"
#include "field.h"
#include "functs.h"
#include "cdo_output.h"


void
Field::init(const CdoVar &var)
{
  grid = var.gridID;
  gridsize = var.gridsize;
  missval = var.missval;
  memType = var.memType;
  size = var.gridsize * var.nwpv;
  m_count = size;
  if (memType == MemType::Float)
    varrayResize(vec_f, size);
  else
    varrayResize(vec_d, size);
}

void
Field::resize(const size_t count)
{
  m_count = count;
  varrayResize(vec_d, m_count);
  if (!size) size = m_count;
}

void
Field::resize(const size_t count, const double value)
{
  m_count = count;
  varrayResizeInit(vec_d, m_count, value);
  if (!size) size = m_count;
}

void
Field::resizef(const size_t count)
{
  m_count = count;
  varrayResize(vec_f, m_count);
  if (!size) size = m_count;
}

void
Field::resizef(const size_t count, const float value)
{
  m_count = count;
  varrayResizeInit(vec_f, m_count, value);
  if (!size) size = m_count;
}

bool
Field::empty() const
{
  return m_count == 0;
}

void
Field::check_gridsize() const
{
  if (size == 0) fprintf(stderr, "Internal problem, size of field not set!\n");
  if (size > m_count) fprintf(stderr, "Internal problem, size of field is greater than allocated size of field!\n");
}

void
Field3D::init(const CdoVar &var)
{
  nlevels = var.nlevels;
  grid = var.gridID;
  gridsize = var.gridsize;
  missval = var.missval;
  memType = var.memType;
  size = var.nlevels * var.gridsize * var.nwpv;
  if (memType == MemType::Float)
    varrayResize(vec_f, size);
  else
    varrayResize(vec_d, size);
}

void
fieldFill(Field &field, double value)
{
  field.check_gridsize();

  if (field.memType == MemType::Float)
    std::fill(field.vec_f.begin(), field.vec_f.begin() + field.size, value);
  else
    std::fill(field.vec_d.begin(), field.vec_d.begin() + field.size, value);
}

void fieldCopy(const Field &field_src, Field &field_tgt)
{
  if (field_src.memType == MemType::Float)
    field_tgt.vec_f = field_src.vec_f;
  else
    field_tgt.vec_d = field_src.vec_d;
}

void fieldCopy(const Field3D &field_src, const int levelID, Field &field_tgt)
{
  const auto size = field_src.gridsize * field_src.nwpv;
  const auto offset = levelID * size;
  if (field_src.memType == MemType::Float)
    std::copy(field_src.vec_f.begin() + offset, field_src.vec_f.begin() + offset + size, field_tgt.vec_f.begin());
  else
    std::copy(field_src.vec_d.begin() + offset, field_src.vec_d.begin() + offset + size, field_tgt.vec_d.begin());
}

void fieldAdd(Field &field1, const Field3D &field2, const int levelID)
{
  const auto size = field1.gridsize * field1.nwpv;
  const auto offset = levelID * size;
  if (field1.memType == MemType::Float)
    for (size_t i = 0; i < size; i++) field1.vec_f[i] += field2.vec_f[offset + i];
  else
    for (size_t i = 0; i < size; i++) field1.vec_d[i] += field2.vec_d[offset + i];

}

// functor that returns true if value is equal to the value of the constructor parameter provided
class valueDblIsEqual
{
  double _missval;

public:
  valueDblIsEqual(double missval) : _missval(missval) {}
  bool
  operator()(const double value) const
  {
    return DBL_IS_EQUAL(value, _missval);
  }
};

// functor that returns true if value is equal to the value of the constructor parameter provided
class valueIsEqual
{
  double _missval;

public:
  valueIsEqual(double missval) : _missval(missval) {}
  bool
  operator()(const double value) const
  {
    return IS_EQUAL(value, _missval);
  }
};

size_t
fieldNumMiss(const Field &field)
{
  field.check_gridsize();

  if (std::isnan(field.missval))
    return std::count_if(field.vec_d.begin(), field.vec_d.begin() + field.size, valueDblIsEqual(field.missval));
  else
    return std::count_if(field.vec_d.begin(), field.vec_d.begin() + field.size, valueIsEqual(field.missval));
}

size_t
fieldNumMV(Field &field)
{
  if (field.memType == MemType::Float)
    field.nmiss = varrayNumMV(field.size, field.vec_f, (float)field.missval);
  else
    field.nmiss = varrayNumMV(field.size, field.vec_d, field.missval);

  return field.nmiss;
}

MinMax
fieldMinMax(const Field &field)
{
  if (field.memType == MemType::Float)
    return varrayMinMax(field.vec_f);
  else
    return varrayMinMax(field.vec_d);
}

double
fieldMin(const Field &field)
{
  if (field.memType == MemType::Float)
    return field.nmiss ? varrayMinMV(field.size, field.vec_f, (float)field.missval) : varrayMin(field.size, field.vec_f);
  else
    return field.nmiss ? varrayMinMV(field.size, field.vec_d, field.missval) : varrayMin(field.size, field.vec_d);
}

double
fieldMax(const Field &field)
{
  if (field.memType == MemType::Float)
    return field.nmiss ? varrayMaxMV(field.size, field.vec_f, (float)field.missval) : varrayMax(field.size, field.vec_f);
  else
    return field.nmiss ? varrayMaxMV(field.size, field.vec_d, field.missval) : varrayMax(field.size, field.vec_d);
}

static double
fieldRange(const Field &field)
{
  if (field.memType == MemType::Float)
    return field.nmiss ? varrayRangeMV(field.size, field.vec_f, (float)field.missval) : varrayRange(field.size, field.vec_f);
  else
    return field.nmiss ? varrayRangeMV(field.size, field.vec_d, field.missval) : varrayRange(field.size, field.vec_d);
}

double
fieldSum(const Field &field)
{
  if (field.memType == MemType::Float)
    return field.nmiss ? varraySumMV(field.size, field.vec_f, (float)field.missval) : varraySum(field.size, field.vec_f);
  else
    return field.nmiss ? varraySumMV(field.size, field.vec_d, field.missval) : varraySum(field.size, field.vec_d);
}

double
fieldMean(const Field &field)
{
  if (field.memType == MemType::Float)
    return field.nmiss ? varrayMeanMV(field.size, field.vec_f, (float)field.missval) : varrayMean(field.size, field.vec_f);
  else
    return field.nmiss ? varrayMeanMV(field.size, field.vec_d, field.missval) : varrayMean(field.size, field.vec_d);
}

double
fieldMeanw(const Field &field)
{
  if (field.memType == MemType::Float)
    return field.nmiss ? varrayWeightedMeanMV(field.size, field.vec_f, field.weightv, (float)field.missval)
                       : varrayWeightedMean(field.size, field.vec_f, field.weightv, (float)field.missval);
  else
    return field.nmiss ? varrayWeightedMeanMV(field.size, field.vec_d, field.weightv, field.missval)
                       : varrayWeightedMean(field.size, field.vec_d, field.weightv, field.missval);
}

double
fieldAvg(const Field &field)
{
  if (field.memType == MemType::Float)
    return field.nmiss ? varrayAvgMV(field.size, field.vec_f, (float)field.missval) : varrayMean(field.size, field.vec_f);
  else
    return field.nmiss ? varrayAvgMV(field.size, field.vec_d, field.missval) : varrayMean(field.size, field.vec_d);
}

double
fieldAvgw(const Field &field)
{
  if (field.memType == MemType::Float)
    return field.nmiss ? varrayWeightedAvgMV(field.size, field.vec_f, field.weightv, (float)field.missval)
                       : varrayWeightedMean(field.size, field.vec_f, field.weightv, (float)field.missval);
  else
    return field.nmiss ? varrayWeightedAvgMV(field.size, field.vec_d, field.weightv, field.missval)
                       : varrayWeightedMean(field.size, field.vec_d, field.weightv, field.missval);
}

double
fieldVar(const Field &field)
{
  if (field.memType == MemType::Float)
    return varrayVar(field.size, field.vec_f, field.nmiss, (float)field.missval);
  else
    return varrayVar(field.size, field.vec_d, field.nmiss, field.missval);
}

double
fieldVar1(const Field &field)
{
  if (field.memType == MemType::Float)
    return varrayVar1(field.size, field.vec_f, field.nmiss, (float)field.missval);
  else
    return varrayVar1(field.size, field.vec_d, field.nmiss, field.missval);
}

static double
fieldKurt(const Field &field)
{
  if (field.memType == MemType::Float)
    return varrayKurt(field.size, field.vec_f, field.nmiss, (float)field.missval);
  else
    return varrayKurt(field.size, field.vec_d, field.nmiss, field.missval);
}

static double
fieldSkew(const Field &field)
{
  if (field.memType == MemType::Float)
    return varraySkew(field.size, field.vec_f, field.nmiss, (float)field.missval);
  else
    return varraySkew(field.size, field.vec_d, field.nmiss, field.missval);
}

double
fieldVarw(const Field &field)
{
  if (field.memType == MemType::Float)
    return varrayWeightedVar(field.size, field.vec_f, field.weightv, field.nmiss, (float)field.missval);
  else
    return varrayWeightedVar(field.size, field.vec_d, field.weightv, field.nmiss, field.missval);
}

double
fieldVar1w(const Field &field)
{
  if (field.memType == MemType::Float)
    return varrayWeightedVar1(field.size, field.vec_f, field.weightv, field.nmiss, (float)field.missval);
  else
    return varrayWeightedVar1(field.size, field.vec_d, field.weightv, field.nmiss, field.missval);
}

double
varToStd(double rvar, double missval)
{
  if (DBL_IS_EQUAL(rvar, missval) || rvar < 0) return missval;

  return IS_NOT_EQUAL(rvar, 0) ? std::sqrt(rvar) : 0;
}

double
fieldStd(const Field &field)
{
  return varToStd(fieldVar(field), field.missval);
}

double
fieldStd1(const Field &field)
{
  return varToStd(fieldVar1(field), field.missval);
}

double
fieldStdw(const Field &field)
{
  return varToStd(fieldVarw(field), field.missval);
}

double
fieldStd1w(const Field &field)
{
  return varToStd(fieldVar1w(field), field.missval);
}

void
vfldrms(const Field &field, const Field &field2, Field &field3)
{
  size_t rnmiss = 0;
  const auto grid1 = field.grid;
  //  size_t nmiss1   = field.nmiss;
  const auto array1 = field.vec_d.data();
  const auto grid2 = field2.grid;
  //  size_t nmiss2   = field2.nmiss;
  const auto array2 = field2.vec_d.data();
  const auto missval1 = field.missval;
  const auto missval2 = field2.missval;
  const auto &w = field.weightv;
  auto rsum = 0.0, rsumw = 0.0;

  const auto len = gridInqSize(grid1);
  if (len != gridInqSize(grid2)) cdoAbort("fields have different size!");

  // if ( nmiss1 )
  {
    for (size_t i = 0; i < len; i++)
      if (!DBL_IS_EQUAL(w[i], missval1))
        {
          rsum = ADDMN(rsum, MULMN(w[i], MULMN(SUBMN(array2[i], array1[i]), SUBMN(array2[i], array1[i]))));
          rsumw = ADDMN(rsumw, w[i]);
        }
  }
  /*
else
  {
    for ( i = 0; i < len; i++ )
      {
        rsum  += w[i] * array1[i];
        rsumw += w[i];
      }
  }
  */

  const auto ravg = SQRTMN(DIVMN(rsum, rsumw));

  if (DBL_IS_EQUAL(ravg, missval1)) rnmiss++;

  field3.vec_d[0] = ravg;
  field3.nmiss = rnmiss;
}

template <typename T>
double
arrayPctl(size_t len, T *array, size_t nmiss, T missval, const double pn)
{
  double pctl = missval;

  if ((len - nmiss) > 0)
    {
      if (nmiss)
        {
          Varray<T> v(len - nmiss);

          size_t j = 0;
          for (size_t i = 0; i < len; i++)
            if (!DBL_IS_EQUAL(array[i], missval)) v[j++] = array[i];

          pctl = percentile(v.data(), j, pn);
        }
      else
        {
          pctl = percentile(array, len, pn);
        }
    }

  return pctl;
}

double
fieldPctl(Field &field, const double pn)
{
  if (field.memType == MemType::Float)
    return arrayPctl(field.size, field.vec_f.data(), field.nmiss, (float)field.missval, pn);
  else
    return arrayPctl(field.size, field.vec_d.data(), field.nmiss, field.missval, pn);
}

static int
compareDouble(const void *const a, const void *const b)
{
  const double *const x = (const double *) a;
  const double *const y = (const double *) b;
  return ((*x > *y) - (*x < *y)) * 2 + (*x > *y) - (*x < *y);
}

double
vfldrank(Field &field)
{
  auto res = 0.0;
  // Using first value as reference (observation)
  const auto array = &field.vec_d[1];
  const auto val = array[-1];
  const auto len = field.size - 1;

  if (field.nmiss) return field.missval;

  std::qsort(array, len, sizeof(double), compareDouble);

  if (val > array[len - 1])
    res = (double) len;
  else
    for (size_t j = 0; j < len; j++)
      if (array[j] >= val)
        {
          res = (double) j;
          break;
        }

  return res;
}

double fldbrs(Field &field)  // Not used!
{
  const auto nmiss = field.nmiss;
  const auto len = field.size;
  const auto array = field.vec_d.data();
  const auto missval = field.missval;

  auto brs = 0.0;
  size_t i, count = 0;

  // Using first value as reference
  if (nmiss == 0)
    {
      for (i = 1; i < len; i++) brs += (array[i] - array[0]) * (array[i] - array[0]);
      count = i - 1;
    }
  else
    {
      if (DBL_IS_EQUAL(array[0], missval)) return missval;

      for (i = 1; i < len; i++)
        if (!DBL_IS_EQUAL(array[i], missval))
          {
            brs += (array[i] - array[0]) * (array[i] - array[0]);
            count++;
          }
    }

  return brs / count;
}

double
fieldFunction(const Field &field, int function)
{
  // clang-format off
  switch (function)
    {
    case func_min:    return fieldMin(field);
    case func_max:    return fieldMax(field);
    case func_range:  return fieldRange(field);
    case func_sum:    return fieldSum(field);
    case func_mean:   return fieldMean(field);
    case func_avg:    return fieldAvg(field);
    case func_std:    return fieldStd(field);
    case func_std1:   return fieldStd1(field);
    case func_var:    return fieldVar(field);
    case func_var1:   return fieldVar1(field);
    case func_meanw:  return fieldMeanw(field);
    case func_avgw:   return fieldAvgw(field);
    case func_stdw:   return fieldStdw(field);
    case func_std1w:  return fieldStd1w(field);
    case func_varw:   return fieldVarw(field);
    case func_var1w:  return fieldVar1w(field);
    case func_kurt:   return fieldKurt(field);
    case func_skew:   return fieldSkew(field);
    default: cdoAbort("%s: function %d not implemented!", __func__, function);
    }
  // clang-format on
  return 0.0;
}
