/*
 *  Licensed to the Apache Software Foundation (ASF) under one or more
 *  contributor license agreements.  See the NOTICE file distributed with
 *  this work for additional information regarding copyright ownership.
 *  The ASF licenses this file to You under the Apache License, Version 2.0
 *  (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 */

package org.apache.tools.ant.taskdefs;

import org.apache.tools.ant.BuildException;
import org.apache.tools.ant.BuildFileRule;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;

import java.util.Arrays;

/**
 */
public class FailTest {

    @Rule
    public ExpectedException thrown = ExpectedException.none();

    @Rule
    public final BuildFileRule buildRule = new BuildFileRule();

    @Before
    public void setUp() {
        buildRule.configureProject("src/etc/testcases/taskdefs/fail.xml");
    }

    @Test
    public void test1() {
        thrown.expect(BuildException.class);
        thrown.expectMessage("No message");
        buildRule.executeTarget("test1");
    }

    @Test
    public void test2() {
        thrown.expect(BuildException.class);
        thrown.expectMessage("test2");
        buildRule.executeTarget("test2");
    }

    @Test
    public void testText() {
        thrown.expect(BuildException.class);
        thrown.expectMessage("testText");
        buildRule.executeTarget("testText");
    }

    @Test(expected = BuildException.class)
    public void testIf() {
        buildRule.executeTarget("testIf");
        buildRule.getProject().setProperty("foo", "");
        buildRule.executeTarget("testIf");
        // TODO assert result
    }

    @Test(expected = BuildException.class)
    public void testUnless() {
        try {
            buildRule.executeTarget("testUnless");
            // TODO assert rules
        } finally {
            buildRule.getProject().setProperty("foo", "");
            buildRule.executeTarget("testUnless");
        }
    }

    /**
     * see that the different combinations work, and
     * that the autogenerated text contains information
     * about which condition was not met
     */
    @Test
    public void testIfAndUnless() {
        thrown.expect(BuildException.class);
        thrown.expectMessage("if=if and unless=unless");
        //neither
        buildRule.executeTarget("testIfAndUnless");
        buildRule.getProject().setProperty("if", "");
        try {
            buildRule.executeTarget("testIfAndUnless");
        } finally {
            buildRule.getProject().setProperty("unless", "");
            //this call should succeed as unless overrides if
            buildRule.executeTarget("testIfAndUnless");
        }
    }
    /**
     * see that the different combinations work, and
     * that the autogenerated text contains information
     * about which condition was not met
     */
    @Test
    public void testIfAndUnless2() {
        buildRule.getProject().setProperty("unless", "");
        buildRule.executeTarget("testIfAndUnless");
    }

    @Test
    public void testNested1() {
        thrown.expect(BuildException.class);
        thrown.expectMessage("condition satisfied");
        buildRule.executeTarget("testNested1");
    }

    @Test
    public void testNested2() {
        buildRule.executeTarget("testNested2");
    }

    @Test
    public void testNested3() {
        thrown.expect(BuildException.class);
        thrown.expectMessage("testNested3");
        buildRule.executeTarget("testNested3");
    }

    @Test
    public void testNested4() {
        thrown.expect(BuildException.class);
        thrown.expectMessage("Nested conditions not permitted in conjunction with if/unless attributes");

        StringBuilder target = new StringBuilder("testNested4x");
        for (char ch : Arrays.asList('a', 'b', 'c')) {
            target.setCharAt(target.length() - 1, ch);
            buildRule.executeTarget(target.toString());
        }
    }

    @Test
    public void testNested5() {
        thrown.expect(BuildException.class);
        thrown.expectMessage("Only one nested condition is allowed.");
        buildRule.executeTarget("testNested5");
    }

    @Test
    public void testNested6() {
        thrown.expect(BuildException.class);
        thrown.expectMessage("testNested6\ntestNested6\ntestNested6");
        buildRule.executeTarget("testNested6");
    }

    @Test
    public void testNested7() {
        thrown.expect(BuildException.class);
        thrown.expectMessage("A single nested condition is required.");

        StringBuilder target = new StringBuilder("testNested7x");
        for (char ch : Arrays.asList('a', 'b')) {
            target.setCharAt(target.length() - 1, ch);
            buildRule.executeTarget(target.toString());
        }
    }

}
