/////////////////////////////////////////////////////////////////////////
// $Id$
/////////////////////////////////////////////////////////////////////////
//
//  Copyright (C) 2012-2024  The Bochs Project
//
//  This library is free software; you can redistribute it and/or
//  modify it under the terms of the GNU Lesser General Public
//  License as published by the Free Software Foundation; either
//  version 2 of the License, or (at your option) any later version.
//
//  This library is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//  Lesser General Public License for more details.
//
//  You should have received a copy of the GNU Lesser General Public
//  License along with this library; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA

#ifndef BX_IODEV_VOODOO_H
#define BX_IODEV_VOODOO_H

#define BX_VOODOO_THIS theVoodooDevice->
#define BX_VOODOO_THIS_PTR theVoodooDevice
#define BX_VVGA_THIS theVoodooVga->
#define BX_VVGA_THIS_PTR theVoodooVga

typedef struct {
  Bit8u model;
  struct {
    Bit32u width;
    Bit32u height;
    Bit64u htotal_usec;
    Bit64u vtotal_usec;
    Bit64u hsync_usec;
    Bit64u vsync_usec;
    double htime_to_pixel;
    Bit64u frame_start;
    bool vfreq_update;
    bool clock_enabled;
    bool output_on;
    bool override_on;
    bool screen_update_pending;
    bool gui_update_pending;
  } vdraw;
  int mode_change_timer_id;
  int vertical_timer_id;
  Bit8u devfunc;
  Bit16u max_xres;
  Bit16u max_yres;
  Bit16u num_x_tiles;
  Bit16u num_y_tiles;
  bool  *vga_tile_updated;
} bx_voodoo_t;


class bx_voodoo_base_c : public bx_nonvga_device_c {
public:
  bx_voodoo_base_c();
  virtual ~bx_voodoo_base_c();
  virtual void init(void);
  virtual void init_model(void) {}
  virtual void register_state(void) {}

  virtual void start_fifo_thread(void);
  virtual void refresh_display(void *this_ptr, bool redraw);
  virtual void redraw_area(unsigned x0, unsigned y0,
                           unsigned width, unsigned height);
  virtual bool update(void);
  virtual bool update_timing(void) {return 0;}
  virtual Bit32u get_retrace(bool hv) {return 0;}
  virtual Bit32u get_vtotal_usec(void) {return 0;}

  virtual void output_enable(bool enabled) {}
  virtual void update_screen_start(void) {}

  virtual void reg_write(Bit32u reg, Bit32u value);
  virtual void blt_reg_write(Bit8u reg, Bit32u value) {}
  virtual void mem_write_linear(Bit32u offset, Bit64u value, unsigned len) {}
  virtual void draw_hwcursor(unsigned xc, unsigned yc, bx_svga_tileinfo_t *info) {}
  virtual void set_tile_updated(unsigned xti, unsigned yti, bool flag) {}

  static void vertical_timer_handler(void *);

protected:
  bx_voodoo_t s;

  void voodoo_register_state(bx_list_c *parent);
  void set_irq_level(bool level);
  void vertical_timer(void);
};

class bx_voodoo_1_2_c : public bx_voodoo_base_c {
public:
  bx_voodoo_1_2_c();
  virtual ~bx_voodoo_1_2_c();
  virtual void init_model(void);
  virtual void reset(unsigned type);
  virtual void register_state(void);
  virtual void after_restore_state(void);

  virtual bool update_timing(void);
  virtual Bit32u get_retrace(bool hv);
  virtual Bit32u get_vtotal_usec(void);

  virtual void output_enable(bool enabled);
  virtual void update_screen_start(void);

  virtual void pci_write_handler(Bit8u address, Bit32u value, unsigned io_len);

private:
  static bool mem_read_handler(bx_phy_address addr, unsigned len, void *data, void *param);
  static bool mem_write_handler(bx_phy_address addr, unsigned len, void *data, void *param);

  void mem_write(bx_phy_address addr, unsigned len, void *data);

  static void mode_change_timer_handler(void *);
  void mode_change_timer(void);
};

class bx_banshee_c : public bx_voodoo_base_c {
public:
  bx_banshee_c();
  virtual ~bx_banshee_c();
  virtual void init_model(void);
  virtual void reset(unsigned type);
  virtual void register_state(void);
  virtual void after_restore_state(void);

  virtual bool   update(void);
  virtual bool   update_timing(void);
  virtual Bit32u get_retrace(bool hv);

  virtual void reg_write(Bit32u reg, Bit32u value);
  virtual void blt_reg_write(Bit8u reg, Bit32u value);
  virtual void mem_write_linear(Bit32u offset, Bit64u value, unsigned len);
  virtual void draw_hwcursor(unsigned xc, unsigned yc, bx_svga_tileinfo_t *info);
  virtual void set_tile_updated(unsigned xti, unsigned yti, bool flag);

  virtual void pci_write_handler(Bit8u address, Bit32u value, unsigned io_len);

  void yuv_planar_write(Bit32u offset, Bit32u value);
  Bit32u blt_reg_read(Bit8u reg);
#if BX_DEBUGGER
  virtual void debug_dump(int argc, char **argv);
#endif

private:
  static bool mem_read_handler(bx_phy_address addr, unsigned len, void *data, void *param);
  static bool mem_write_handler(bx_phy_address addr, unsigned len, void *data, void *param);

  void mem_read(bx_phy_address addr, unsigned len, void *data);
  void mem_write(bx_phy_address addr, unsigned len, void *data);

  static Bit32u read_handler(void *this_ptr, Bit32u address, unsigned io_len);
  static void   write_handler(void *this_ptr, Bit32u address, Bit32u value, unsigned io_len);

  Bit32u read(Bit32u address, unsigned io_len);
  void   write(Bit32u address, Bit32u value, unsigned io_len);

  Bit32u agp_reg_read(Bit8u reg);
  void   agp_reg_write(Bit8u reg, Bit32u value);

  void   blt_launch_area_setup(void);
  void   blt_launch_area_write(Bit32u value);
  void   blt_execute(void);
  void   blt_complete(void);
  bool   blt_apply_clipwindow(int *x0, int *y0, int *x1, int *y1, int *w, int *h);
  bool   blt_clip_check(int x, int y);
  Bit8u  blt_colorkey_check(Bit8u *ptr, Bit8u pxsize, bool dst);
  Bit32u blt_yuv_conversion(Bit8u *ptr, Bit16u xc, Bit16u yc, Bit16u pitch, Bit8u fmt, Bit8u pxsize);

  void   blt_rectangle_fill(void);
  void   blt_pattern_fill_mono(void);
  void   blt_pattern_fill_color(void);
  void   blt_screen_to_screen(void);
  void   blt_screen_to_screen_pattern(void);
  void   blt_screen_to_screen_stretch(void);
  void   blt_host_to_screen(void);
  void   blt_host_to_screen_stretch(void);
  void   blt_host_to_screen_pattern(void);
  void   blt_line(bool pline);
  void   blt_polygon_fill(bool force);

  Bit32u get_overlay_pixel(unsigned x, unsigned y, Bit8u bpp);
  bool   chromakey_check(Bit32u color, Bit8u bpp);

  bx_ddc_c ddc;
  bool     is_agp;
};

class bx_voodoo_vga_c : public bx_vgacore_c {
public:
  bx_voodoo_vga_c();
  virtual ~bx_voodoo_vga_c();

  virtual void   reset(unsigned type);
  virtual void   register_state(void);
  virtual void   after_restore_state(void);

  virtual Bit8u  mem_read(bx_phy_address addr);
  virtual void   mem_write(bx_phy_address addr, Bit8u value);

  virtual void   redraw_area(unsigned x0, unsigned y0,
                             unsigned width, unsigned height);

  virtual bool   init_vga_extension(void);
  virtual void   get_crtc_params(bx_crtc_params_t *crtcp, Bit32u *vclock);
  Bit32u get_retrace(void);

  void banshee_update_mode(void);
  void banshee_set_dac_mode(bool mode);
  void banshee_set_vclk3(Bit32u value);

  static Bit32u banshee_vga_read_handler(void *this_ptr, Bit32u address, unsigned io_len);
  static void   banshee_vga_write_handler(void *this_ptr, Bit32u address, Bit32u value, unsigned io_len);

protected:
  virtual void  update(void);
};

extern bx_voodoo_base_c* theVoodooDevice;
extern bx_voodoo_vga_c* theVoodooVga;

#endif
