<?hh
// This doc comment block generated by idl/sysdoc.php
/**
 * ( excerpt from http://php.net/manual/en/class.phar.php )
 *
 * The Phar class provides a high-level interface to accessing and
 * creating phar archives.
 *
 */
class Phar extends RecursiveDirectoryIterator
  implements Countable, ArrayAccess {

  const NONE = 0;
  const COMPRESSED = 0x0000F000;
  const GZ = 0x00001000;
  const BZ2 = 0x00002000;
  const SIGNATURE = 0x00010000;

  const SAME = 0;
  const PHAR = 1;
  const TAR = 2;
  const ZIP = 3;

  const MD5 = 0x0001;
  const SHA1 = 0x0002;
  const SHA256 = 0x0003;
  const SHA512 = 0x0004;
  const OPENSSL = 0x0010;

  const PHP = 1;
  const PHPS = 2;

  const HALT_TOKEN = '__HALT_COMPILER();';

  /**
   * A map from filename_or_alias => Phar object
   */
  private static $aliases = array();
  /**
   * Prevent the check for __HALT_COMPILER()
   */
  private static $preventHaltTokenCheck = false;
  /**
   * Prevent the check for file extension
   */
  private static $preventExtCheck = false;

  protected $fname;
  /**
   * @var __SystemLib\ArchiveHandler
   */
  protected $archiveHandler;
  protected $iteratorRoot;
  protected $iterator;
  protected int $format = self::PHAR;

  /**
   * ( excerpt from http://php.net/manual/en/phar.construct.php )
   *
   *
   * @param string $fname Path to an existing Phar archive or to-be-created
   *                      archive. The file name's extension must contain
   *                      .phar.
   * @param int $flags    Flags to pass to parent class
   *                      RecursiveDirectoryIterator.
   * @param string $alias Alias with which this Phar archive should be referred
   *                      to in calls to stream functionality.
   *
   * @throws PharException
   * @throws UnexpectedValueException
   */
  public function __construct($fname, $flags = null, $alias = null) {
    if (!self::$preventExtCheck && !self::isValidPharFilename($fname)) {
      throw new UnexpectedValueException(
        "Cannot create phar '$fname', file extension (or combination) not".
        ' recognised or the directory does not exist'
      );
    }
    if (!is_file($fname)) {
      throw new UnexpectedValueException("$fname is not a file");
    }
    $this->fname = $fname;
    $fp = fopen($fname, 'rb');
    $this->iteratorRoot = 'phar://'.realpath($fname).'/';

    $magic_number = fread($fp, 4);
    // This is not a bullet-proof check, but should be good enough to catch ZIP
    if (strcmp($magic_number, "PK\x03\x04") === 0) {
      fclose($fp);
      $this->format = self::ZIP;
      $this->archiveHandler = new __SystemLib\ZipArchiveHandler(
        $fname,
        self::$preventHaltTokenCheck
      );
    } else {
      if (strpos($magic_number, 'BZ') === 0) {
        fclose($fp);
        $fp = bzopen($fname, 'r');
        fseek($fp, 257, SEEK_CUR); // Bzip2 only knows how to use SEEK_CUR
      } else if (strpos($magic_number, "\x1F\x8B") === 0) {
        fclose($fp);
        $fp = gzopen($fname, 'rb');
        fseek($fp, 257);
      } else {
        fseek($fp, 257);
      }
      $magic_number = fread($fp, 8);
      fclose($fp);
      if (
        strpos($magic_number, "ustar\x0") === 0 ||
        strpos($magic_number, "ustar\x40\x40\x0") === 0
      ) {
        $this->format = self::TAR;
        $this->archiveHandler = new __SystemLib\TarArchiveHandler(
          $fname,
          self::$preventHaltTokenCheck
        );
      } else {
        $this->format = self::PHAR;
        $this->archiveHandler = new __SystemLib\PharArchiveHandler(
          $fname,
          self::$preventHaltTokenCheck
        );
      }
    }
    $aliasFromManifest = $this->getAlias();
    if ($aliasFromManifest) {
      $this->assertCorrectAlias($aliasFromManifest);
      self::$aliases[$aliasFromManifest] = $this;
    }
    if ($alias) {
      $this->assertCorrectAlias($alias);
      self::$aliases[$alias] = $this;
    }
    // We also do filename lookups
    self::$aliases[$fname] = $this;
  }

  /**
   * ( excerpt from http://php.net/manual/en/phar.addemptydir.php )
   *
   *
   * @param string $dirname The name of the empty directory to create in the
   *                        phar archive
   * @param int $levels     The number of parent directories to go up. This
   *                        must be an integer greater than 0.
   *
   * @return void no return value, exception is thrown on failure.
   */
  public function addEmptyDir($dirname, $levels = 1) {
    throw new UnexpectedValueException('phar is read-only');
  }

  /**
   * ( excerpt from http://php.net/manual/en/phar.addfile.php )
   *
   *
   * @param string $file      Full or relative path to a file on disk to be
   *                          added to the phar archive.
   * @param string $localname Path that the file will be stored in the archive.
   *
   * @return void no return value, exception is thrown on failure.
   */
  public function addFile($file, $localname = null) {
    throw new UnexpectedValueException('phar is read-only');
  }

  /**
   * ( excerpt from http://php.net/manual/en/phar.addfromstring.php )
   *
   *
   * @param string $localname Path that the file will be stored in the archive.
   * @param string $contents  The file contents to store
   *
   * @return void no return value, exception is thrown on failure.
   */
  public function addFromString($localname, $contents) {
    throw new UnexpectedValueException('phar is read-only');
  }

  /**
   * ( excerpt from http://php.net/manual/en/phar.buildfromdirectory.php )
   *
   *
   * @param string $base_dir The full or relative path to the directory that
   *                         contains all files to add to the archive.
   * @param string $regex    An optional pcre regular expression that is used
   *                         to filter the list of files. Only file paths
   *                         matching the regular expression will be included
   *                         in the archive.
   *
   * @return array <b>Phar::buildFromDirectory</b> returns an associative array
   */
  public function buildFromDirectory($base_dir, $regex = null) {
    throw new UnexpectedValueException('phar is read-only');
  }

  /**
   * ( excerpt from http://php.net/manual/en/phar.buildfromiterator.php )
   *
   *
   * @param Iterator $iter         Any iterator that either associatively maps
   *                               phar file to location or returns SplFileInfo
   *                               objects
   * @param string $base_directory For iterators that return SplFileInfo
   *                               objects, the portion of each file's full
   *                               path to remove when adding to the phar
   *                               archive
   *
   * @return array <b>Phar::buildFromIterator</b> returns an associative array
   *                               mapping internal path of file to the full
   *                               path of the file on the filesystem.
   */
  public function buildFromIterator($iter, $base_directory = null) {
    throw new UnexpectedValueException('phar is read-only');
  }

  /**
   * ( excerpt from http://php.net/manual/en/phar.compressfiles.php )
   *
   *
   * @param int $compression Compression must be one of Phar::GZ,
   *                         Phar::BZ2 to add compression, or Phar::NONE
   *                         to remove compression.
   *
   * @return void No value is returned.
   */
  public function compressFiles($compression) {
    throw new UnexpectedValueException('phar is read-only');
  }

  /**
   * ( excerpt from http://php.net/manual/en/phar.decompressfiles.php )
   *
   *
   * @return bool <b>TRUE</b> on success or <b>FALSE</b> on failure.
   */
  public function decompressFiles() {
    throw new UnexpectedValueException('phar is read-only');
  }

  /**
   * ( excerpt from http://php.net/manual/en/phar.compress.php )
   *
   *
   * @param int $compression  Compression must be one of Phar::GZ,
   *                          Phar::BZ2 to add compression, or Phar::NONE
   *                          to remove compression.
   * @param string $extension By default, the extension is .phar.gz
   *                          or .phar.bz2 for compressing phar archives, and
   *                          .phar.tar.gz or .phar.tar.bz2 for
   *                          compressing tar archives. For decompressing, the
   *                          default file extensions are .phar and .phar.tar.
   *
   * @return object a <b>Phar</b> object.
   */
  public function compress($compression, $extension = null) {
    throw new UnexpectedValueException('phar is read-only');
  }

  /**
   * ( excerpt from http://php.net/manual/en/phar.decompress.php )
   *
   *
   * @param string $extension For decompressing, the default file extensions
   *                          are .phar and .phar.tar.
   *                          Use this parameter to specify another file
   *                          extension. Be aware that all executable phar
   *                          archives must contain .phar in their filename.
   *
   * @return object A <b>Phar</b> object is returned.
   */
  public function decompress($extension = null) {
    throw new UnexpectedValueException('phar is read-only');
  }

  /**
   * ( excerpt from http://php.net/manual/en/phar.converttoexecutable.php )
   *
   *
   * @param int $format       This should be one of Phar::PHAR, Phar::TAR,
   *                          or Phar::ZIP. If set to <b>NULL</b>, the existing
   *                          file format will be preserved.
   * @param int $compression  This should be one of Phar::NONE for no
   *                          whole-archive compression, Phar::GZ for
   *                          zlib-based compression, and Phar::BZ2 for
   *                          bzip-based compression.
   * @param string $extension This parameter is used to override the default
   *                          file extension for a converted archive. Note that
   *                          all zip- and tar-based phar archives must contain
   *                          .phar in their file extension in order to be
   *                          processed as a phar archive.
   *
   *                          If converting to a phar-based archive, the
   *                          default extensions are
   *                          .phar, .phar.gz, or .phar.bz2
   *                          depending on the specified compression. For
   *                          tar-based phar archives, the default extensions
   *                          are .phar.tar, .phar.tar.gz, and .phar.tar.bz2.
   *                          For zip-based phar archives, the default
   *                          extension is .phar.zip.
   *
   * @return Phar The method returns a <b>Phar</b> object on success and throws
   *              an exception on failure.
   */
  public function convertToExecutable($format = 9021976,
                                      $compression_type = 9021976,
                                      $file_ext = null) {
    throw new UnexpectedValueException('phar is read-only');
  }

  /**
   * ( excerpt from http://php.net/manual/en/phar.converttodata.php )
   *
   * This method is used to convert an executable phar archive to either a
   * tar or zip file. To make the tar or zip non-executable, the phar stub
   * and phar alias files are removed from the newly created archive.
   *
   * If no changes are specified, this method throws a
   * BadMethodCallException if the archive is in phar file format. For
   * archives in tar or zip file format, this method converts the archive to
   * a non-executable archive.
   *
   * If successful, the method creates a new archive on disk and returns a
   * PharData object. The old archive is not removed from disk, and should be
   * done manually after the process has finished.
   *
   * @param int $format       This should be one of Phar::TAR
   *                          or Phar::ZIP. If set to <b>NULL</b>, the existing
   *                          file format will be preserved.
   * @param int $compression  This should be one of Phar::NONE for no
   *                          whole-archive compression, Phar::GZ for
   *                          zlib-based compression, and Phar::BZ2 for
   *                          bzip-based compression.
   * @param string $extension This parameter is used to override the default
   *                          file extension for a converted archive. Note that
   *                          .phar cannot be used anywhere in the filename for
   *                          a non-executable tar or zip archive.
   *                          </p>
   *                          If converting to a tar-based phar archive, the
   *                          default extensions are .tar, .tar.gz,
   *                          and .tar.bz2 depending on specified compression.
   *                          For zip-based archives, the
   *                          default extension is .zip.
   *
   * @return PharData The method returns a <b>PharData</b> object on success
   *                  and throws an exception on failure.
   */
  public function convertToData($format = 9021976,
                                $compression_type = 9021976,
                                $extension = null) {
    throw new UnexpectedValueException('phar is read-only');
  }

  /**
   * ( excerpt from http://php.net/manual/en/phar.copy.php )
   *
   *
   * @param string $oldfile
   * @param string $newfile
   * @return bool returns <b>TRUE</b> on success, but it is safer to encase
   *              method call in a try/catch block and assume success if no
   *              exception is thrown.
   */
  public function copy($oldfile, $newfile) {
    throw new UnexpectedValueException('phar is read-only');
  }

  /**
   * ( excerpt from http://php.net/manual/en/phar.count.php )
   *
   *
   * @return int The number of files contained within this phar, or 0 (the
   *             number zero) if none.
   */
  public function count() {
    return $this->archiveHandler->count();
  }

  /**
   * ( excerpt from http://php.net/manual/en/phar.delete.php )
   *
   *
   * @param string $entry Path within an archive to the file to delete.
   *
   * @return bool returns <b>TRUE</b> on success, but it is better to check for
   *              thrown exception, and assume success if none is thrown.
   */
  public function delete($entry) {
    throw new UnexpectedValueException('phar is read-only');
  }

  /**
   * ( excerpt from http://php.net/manual/en/phar.delmetadata.php )
   *
   *
   * @return bool returns <b>TRUE</b> on success, but it is better to check for
   *              thrown exception, and assume success if none is thrown.
   */
  public function delMetadata() {
    throw new UnexpectedValueException('phar is read-only');
  }

  /**
   * ( excerpt from http://php.net/manual/en/phar.extractto.php )
   *
   *
   * @param string $pathto      Path within an archive to the file to delete.
   * @param string|array $files The name of a file or directory to extract, or
   *                            an array of files/directories to extract
   * @param bool $overwrite     Set to <b>TRUE</b> to enable overwriting
   *                            existing files
   *
   * @return bool returns <b>TRUE</b> on success, but it is better to check for
   *              thrown exception, and assume success if none is thrown.
   */
  public function extractTo($pathto, $files = null, $overwrite = false) {
    throw new UnexpectedValueException('phar is read-only');
  }

  public function getAlias() {
    return $this->archiveHandler->getAlias();
  }

  public function getPath() {
    return $this->path;
  }

  /**
   * ( excerpt from http://php.net/manual/en/phar.getmetadata.php )
   *
   * Retrieve archive meta-data. Meta-data can be any PHP variable that can
   * be serialized.
   * No parameters.
   *
   * @return mixed any PHP variable that can be serialized and is stored as
   *               meta-data for the Phar archive, or <b>NULL</b> if no
   *               meta-data is stored.
   */
  public function getMetadata() {
    return $this->archiveHandler->getMetadata();
  }

  /**
   * ( excerpt from http://php.net/manual/en/phar.getmodified.php )
   *
   * This method can be used to determine whether a phar has either had an
   * internal file deleted, or contents of a file changed in some way.
   * No parameters.
   *
   * @return bool <b>TRUE</b> if the phar has been modified since opened,
   *              <b>FALSE</b> if not.
   */
  public function getModified() {
    return false;
  }

  /**
   * ( excerpt from http://php.net/manual/en/phar.getsignature.php )
   *
   * Returns the verification signature of a phar archive in a hexadecimal
   * string.
   *
   * @return array Array with the opened archive's signature in hash key and
   *               MD5, SHA-1, SHA-256, SHA-512, or OpenSSL in hash_type. This
   *               signature is a hash calculated on the entire phar's
   *               contents, and may be used to verify the integrity of the
   *               archive. A valid signature is absolutely required of all
   *               executable phar archives if the phar.require_hash INI
   *               variable is set to true.
   */
  public function getSignature() {
    return $this->archiveHandler->getSignature();
  }

  /**
   * ( excerpt from http://php.net/manual/en/phar.getstub.php )
   *
   * Phar archives contain a bootstrap loader, or stub written in PHP that
   * is executed when the archive is executed in PHP either via include:
   *
   * or by simple execution: php myphar.phar
   *
   * @return string a string containing the contents of the bootstrap loader
   *                (stub) of the current Phar archive.
   */
  public function getStub() {
    return $this->archiveHandler->getStub();
  }

  /**
   * ( excerpt from http://php.net/manual/en/phar.getversion.php )
   *
   * Returns the API version of an opened Phar archive.
   *
   * @return string The opened archive's API version. This is not to be
   *                confused with the API version that the loaded phar
   *                extension will use to create new phars. Each Phar archive
   *                has the API version hard-coded into its manifest. See Phar
   *                file format documentation for more information.
   */
  public function getVersion() {
    return $this->archiveHandler->apiVersion();
  }

  /**
   * ( excerpt from http://php.net/manual/en/phar.hasmetadata.php )
   *
   * Returns whether phar has global meta-data set.
   * No parameters.
   *
   * @return bool <b>TRUE</b> if meta-data has been set, and <b>FALSE</b> if
   *              not.
   */
  public function hasMetadata() {
    return $this->archiveHandler->hasMetadata();
  }

  /**
   * ( excerpt from http://php.net/manual/en/phar.isbuffering.php )
   *
   * This method can be used to determine whether a Phar will save changes
   * to disk immediately, or whether a call to Phar::stopBuffering() is
   * needed to enable saving changes.
   *
   * Phar write buffering is per-archive, buffering active for the foo.phar
   * Phar archive does not affect changes to the bar.phar Phar archive.
   *
   * @return bool <b>TRUE</b> if the write operations are being buffer,
   *              <b>FALSE</b> otherwise.
   */
  public function isBuffering() {
    return false;
  }

  /**
   * ( excerpt from http://php.net/manual/en/phar.iscompressed.php )
   *
   * No parameters.
   *
   *
   * @return mixed Phar::GZ, Phar::BZ2 or <b>FALSE</b>
   */
  public function isCompressed() {
    return $this->archiveHandler->isCompressed();
  }

  /**
   * ( excerpt from http://php.net/manual/en/phar.isfileformat.php )
   *
   *
   * @param int $format Either Phar::PHAR, Phar::TAR, or
   *                    Phar::ZIP to test for the format of the archive.
   *
   * @return bool <b>TRUE</b> if the phar archive matches the file format
   *              requested by the parameter
   */
  public function isFileFormat($fileformat) {
    return $fileformat === $this->format;
  }

  /**
   * ( excerpt from http://php.net/manual/en/phar.isvalidpharfilename.php )
   *
   *
   * @param string $filename The name or full path to a phar archive not yet
   *                         created.
   * @param bool $executable This parameter determines whether the filename
   *                         should be treated as a phar executable archive, or
   *                         a data non-executable archive.
   *
   * @return bool <b>TRUE</b> if the filename is valid, <b>FALSE</b> if not.
   */
  public static function isValidPharFilename(
    string $filename,
    bool $executable = true
  ): bool {
    $filename = basename($filename);
    $pharExt = preg_match('/.+\.phar(\..+|$)/i', $filename) === 1;
    return $executable ? $pharExt
                       : !$pharExt && strpos($filename, '.') !== false;
  }

  /**
   * ( excerpt from http://php.net/manual/en/phar.iswritable.php )
   *
   * This method returns TRUE if phar.readonly is 0, and the actual phar
   * archive on disk is not read-only.
   * No parameters.
   *
   * @return bool <b>TRUE</b> if the phar archive can be modified
   */
  public function isWritable() {
    return false;
  }

  /**
   * ( excerpt from http://php.net/manual/en/phar.offsetexists.php )
   *
   * This is an implementation of the ArrayAccess interface allowing direct
   * manipulation of the contents of a Phar archive using array access
   * brackets.
   *
   * @param string $offset The filename (relative path) to look for in a Phar.
   *
   * @return bool <b>TRUE</b> if the file exists within the phar, or
   *              <b>FALSE</b> if not.
   */
  public function offsetExists($offset) {
    return $this->archiveHandler->getEntriesMap()->containsKey($offset);
  }

  /**
   * ( excerpt from http://php.net/manual/en/phar.offsetget.php )
   *
   * This is an implementation of the ArrayAccess interface allowing direct
   * manipulation of the contents of a Phar archive using array access
   * brackets. Phar::offsetGet() is used for retrieving files from a Phar
   * archive.
   *
   * @param string $offset The filename (relative path) to look for in a Phar.
   *
   * @return int A <b>PharFileInfo</b> object is returned that can be used to
   *                       iterate over a file's contents or to retrieve
   *                       information about the current file.
   */
  public function offsetGet($offset) {
    $entry = $this->archiveHandler->getEntriesMap()->get($offset);
    if (!$entry) {
      return null;
    }
    return new PharFileInfo($this->iteratorRoot.$offset, $entry);
  }

  /**
   * ( excerpt from http://php.net/manual/en/phar.offsetset.php )
   *
   *
   * @param string $offset The filename (relative path) to modify in a Phar.
   * @param string $value  Content of the file.
   *
   * @return void No return values.
   */
  public function offsetSet($offset, $value) {
    throw new Exception('Not implemented yet');
  }

  /**
   * ( excerpt from http://php.net/manual/en/phar.offsetunset.php )
   *
   *
   * @param string $offset The filename (relative path) to modify in a Phar.
   *
   * @return bool <b>TRUE</b> on success or <b>FALSE</b> on failure.
   */
  public function offsetUnset($offset) {
    throw new UnexpectedValueException('phar is read-only');
  }

  /**
   * ( excerpt from http://php.net/manual/en/phar.setalias.php )
   *
   *
   * @param string $alias A shorthand string that this archive can be referred
   *                      to in phar stream wrapper access.
   *
   * @return bool
   */
  public function setAlias($alias) {
    $this->assertCorrectAlias($alias);
    if (isset(self::$aliases[$alias])) {
      $path = self::$aliases[$alias]->fname;
      throw new PharException(
        "alias \"$alias\" is already used for archive \"$path\" and cannot".
        ' be used for other archives'
      );
    }
    // TODO: Remove following line when write support implemented
    throw new UnexpectedValueException('phar is read-only');
    self::assertWriteSupport();
    $this->archiveHandler->setAlias($alias);
  }

  private function assertCorrectAlias(string $alias) {
    if (preg_match('#[\\/:;]#', $alias)) {
      throw new UnexpectedValueException(
        "Invalid alias \"$alias\" specified for phar \"$this->fname\""
      );
    }
  }

  /**
   * ( excerpt from http://php.net/manual/en/phar.setdefaultstub.php )
   *
   *
   * @param string $index    Relative path within the phar archive to run if
   *                         accessed on the command-line
   * @param string $webindex Relative path within the phar archive to run if
   *                         accessed through a web browser
   *
   * @return bool <b>TRUE</b> on success or <b>FALSE</b> on failure.
   */
  public function setDefaultStub($index, $webindex = null) {
    throw new UnexpectedValueException('phar is read-only');
  }

  /**
   * ( excerpt from http://php.net/manual/en/phar.setmetadata.php )
   *
   *
   * @param mixed $metadata Any PHP variable containing information to store
   *                        that describes the phar archive
   *
   * @return void No value is returned.
   */
  public function setMetadata($metadata) {
    throw new UnexpectedValueException('phar is read-only');
  }

  /**
   * ( excerpt from http://php.net/manual/en/phar.setsignaturealgorithm.php )
   *
   *
   * @param int $sigtype       One of Phar::MD5,
   *                           Phar::SHA1, Phar::SHA256,
   *                           Phar::SHA512, or Phar::OPENSSL
   * @param string $privatekey The contents of an OpenSSL private key, as
   *                           extracted from a certificate or OpenSSL key
   *                           file:
   *                           <code>
   *                           $private =
   *                           openssl_get_privatekey(file_get_contents('private.pem'));
   *                           $pkey = '';
   *                           openssl_pkey_export($private, $pkey);
   *                           $p->setSignatureAlgorithm(Phar::OPENSSL, $pkey);
   *                           </code>
   *                           See phar introduction for instructions on
   *                           naming and placement of the public key file.
   *
   * @return void No value is returned.
   */
  public function setSignatureAlgorithm($sigtype, $privatekey = null) {
    throw new UnexpectedValueException('phar is read-only');
  }

  /**
   * ( excerpt from http://php.net/manual/en/phar.setstub.php )
   *
   *
   * @param resource|string $stub A string or an open stream handle to use as
   *                              the executable stub for this phar archive.
   * @param int $len
   *
   * @return bool <b>TRUE</b> on success or <b>FALSE</b> on failure.
   */
  public function setStub($stub, $len = -1) {
    // TODO: Remove following line when write support implemented
    throw new UnexpectedValueException('phar is read-only');
    self::assertWriteSupport();
    if (is_resource($stub)) {
      $stub = stream_get_contents($stub);
    }
    $this->archiveHandler->setStub($stub, $len);
  }

  /**
   * ( excerpt from http://php.net/manual/en/phar.startbuffering.php )
   *
   * Although technically unnecessary, the Phar::startBuffering() method can
   * provide a significant performance boost when creating or modifying a
   * Phar archive with a large number of files. Ordinarily, every time a file
   * within a Phar archive is created or modified in any way, the entire Phar
   * archive will be recreated with the changes. In this way, the archive
   * will be up-to-date with the activity performed on it.
   *
   * However, this can be unnecessary when simply creating a new Phar
   * archive, when it would make more sense to write the entire archive out
   * at once. Similarly, it is often necessary to make a series of changes
   * and to ensure that they all are possible before making any changes on
   * disk, similar to the relational database concept of transactions. the
   * Phar::startBuffering()/ Phar::stopBuffering() pair of methods is
   * provided for this purpose.
   *
   * Phar write buffering is per-archive, buffering active for the foo.phar
   * Phar archive does not affect changes to the bar.phar Phar archive.
   *
   * @return void No value is returned.
   */
  public function startBuffering() {
    throw new UnexpectedValueException('phar is read-only');
  }

  /**
   * ( excerpt from http://php.net/manual/en/phar.stopbuffering.php )
   *
   * Phar::stopBuffering() is used in conjunction with the
   * Phar::startBuffering() method. Phar::startBuffering() can provide a
   * significant performance boost when creating or modifying a Phar archive
   * with a large number of files. Ordinarily, every time a file within a
   * Phar archive is created or modified in any way, the entire Phar archive
   * will be recreated with the changes. In this way, the archive will be
   * up-to-date with the activity performed on it.
   *
   * However, this can be unnecessary when simply creating a new Phar
   * archive, when it would make more sense to write the entire archive out
   * at once. Similarly, it is often necessary to make a series of changes
   * and to ensure that they all are possible before making any changes on
   * disk, similar to the relational database concept of transactions. The
   * Phar::startBuffering()/ Phar::stopBuffering() pair of methods is
   * provided for this purpose.
   *
   * Phar write buffering is per-archive, buffering active for the foo.phar
   * Phar archive does not affect changes to the bar.phar Phar archive.
   *
   * @return void No value is returned.
   */
  public function stopBuffering() {
    throw new UnexpectedValueException('phar is read-only');
  }

  /**
   * ( excerpt from http://php.net/manual/en/phar.canwrite.php )
   *
   * This static method determines whether write access has been disabled in
   * the system php.ini via the phar.readonly ini variable.
   *
   * @return bool <b>TRUE</b> if write access is enabled, <b>FALSE</b> if it is
   *              disabled.
   */
  public static function canWrite()  {
    return false;
  }

  /**
   * ( excerpt from http://php.net/manual/en/phar.apiversion.php )
   *
   * Return the API version of the phar file format that will be used when
   * creating phars. The Phar extension supports reading API version 1.0.0 or
   * newer. API version 1.1.0 is required for SHA-256 and SHA-512 hash, and
   * API version 1.1.1 is required to store empty directories.
   *
   * @return string The API version string as in "1.0.0".
   */
  final public static function apiVersion() {
    return '1.0.0';
  }

  /**
   * ( excerpt from http://php.net/manual/en/phar.cancompress.php )
   *
   * This should be used to test whether compression is possible prior to
   * loading a phar archive containing compressed files.
   *
   * @param int $type Either Phar::GZ or Phar::BZ2 can be
   *                  used to test whether compression is possible with a
   *                  specific compression algorithm (zlib or bzip2).
   *
   * @return bool <b>TRUE</b> if compression/decompression is available,
   *              <b>FALSE</b> if not.
   */
  final public static function canCompress($type = 0) {
    return false;
  }

  /**
   * ( excerpt from http://php.net/manual/en/phar.getsupportedcompression.php
   * )
   *
   *
   * No parameters.
   *
   * @return array an array containing any of Phar::GZ or Phar::BZ2, depending
   *               on the availability of the zlib extension or the bz2
   *               extension.
   */
  final public static function getSupportedCompression() {
    return [self::GZ, self::BZ2];
  }

  /**
   * ( excerpt from http://php.net/manual/en/phar.getsupportedsignatures.php
   * )
   *
   * Return array of supported signature types
   * No parameters.
   *
   * @return array an array containing any of MD5, SHA-1, SHA-256, SHA-512, or
   *               OpenSSL.
   */
  final public static function getSupportedSignatures () {
    return ['MD5', 'SHA-1', 'SHA-256', 'SHA-512'];
  }

  /**
   * ( excerpt from http://php.net/manual/en/phar.loadphar.php )
   *
   * This can be used to read the contents of an external Phar archive. This
   * is most useful for assigning an alias to a phar so that subsequent
   * references to the phar can use the shorter alias, or for loading Phar
   * archives that only contain data and are not intended for
   * execution/inclusion in PHP scripts.
   *
   * @param string $filename the full or relative path to the phar archive to
   *                         open
   * @param string $alias    The alias that may be used to refer to the phar
   *                         archive. Note that many phar archives specify an
   *                         explicit alias inside the phar archive, and a
   *                         <b>PharException</b> will be thrown if a new alias
   *                         is specified in this case.
   *
   * @return bool <b>TRUE</b> on success or <b>FALSE</b> on failure.
   */
  final public static function loadPhar($filename, $alias = null) {
    // We need this hack because the stream wrapper should work
    // even without the __HALT_COMPILER token
    self::$preventHaltTokenCheck = true;
    new self($filename, null, $alias);
    self::$preventHaltTokenCheck = false;
    return true;
  }

  /**
   * ( excerpt from http://php.net/manual/en/phar.mapphar.php )
   *
   * This static method can only be used inside a Phar archive's loader stub
   * in order to initialize the phar when it is directly executed, or when it
   * is included in another script.
   *
   * @param string $alias   The alias that can be used in phar:// URLs to
   *                        refer to this archive, rather than its full path.
   * @param int $dataoffset Unused variable, here for compatibility with PEAR's
   *                        PHP_Archive.
   *
   * @return bool <b>TRUE</b> on success or <b>FALSE</b> on failure.
   */
  public static function mapPhar($alias = null, $dataoffset = 0) {
    // We need this hack because extension check during mapping is not needed
    self::$preventExtCheck = true;
    new self(debug_backtrace()[0]['file'], null, $alias);
    self::$preventExtCheck = false;
    return true;
  }

  /**
   * ( excerpt from http://php.net/manual/en/phar.interceptfilefuncs.php )
   *
   * instructs phar to intercept fopen(), readfile(), file_get_contents(),
   * opendir(), and all of the stat-related functions. If any of these
   * functions is called from within a phar archive with a relative path, the
   * call is modified to access a file within the phar archive. Absolute
   * paths are assumed to be attempts to load external files from the
   * filesystem.
   *
   * This function makes it possible to run PHP applications designed to run
   * off of a hard disk as a phar application.
   * No parameters.
   *
   * @return void
   */
  public static function interceptFileFuncs() {
    // Not supported (yet) but most phars call it, so don't throw
  }

  /**
   * ( excerpt from http://php.net/manual/en/phar.running.php )
   *
   * Returns the full path to the running phar archive. This is intended
   * for use much like the __FILE__ magic constant, and only has effect
   * inside an executing phar archive.
   *
   * Inside the stub of an archive, function Phar::running() returns "".
   * Simply use __FILE__ to access the current running phar inside a stub
   *
   * @param bool $retphar If <b>FALSE</b>, the full path on disk to the phar
   *                      archive is returned. If <b>TRUE</b>, a full phar URL
   *                      is returned.
   *
   * @return string the filename if valid, empty string otherwise.
   */
  final public static function running(bool $retphar = true) {
    $filename = debug_backtrace(DEBUG_BACKTRACE_IGNORE_ARGS)[0]['file'];
    $pharScheme = 'phar://';
    $pharExt = '.phar';
    if (strpos($filename, $pharScheme) === 0) {
      $pharExtPos = strrpos($filename, $pharExt);
      if ($pharExtPos) {
        $endPos = $pharExtPos + strlen($pharExt);
        if ($retphar) {
          return substr($filename, 0, $endPos);
        } else {
          return substr($filename, strlen($pharScheme),
            $endPos - strlen($pharScheme));
        }
      }
    }
    return '';
  }

  final public static function webPhar(
      $alias,
      $index = 'index.php',
      $f404 = null,
      $mimetypes = null,
      $rewrites = null) {
    // This is in the default stub, but lets ignore it for now
  }

  /**
   * A poor man's FileUtil::canonicalize in PHP
   */
  private static function resolveDotDots($pieces) {
    $starts_with_slash = false;
    if (count($pieces) > 0 && !strlen($pieces[0])) {
      $starts_with_slash = true;
    }

    foreach ($pieces as $i => $piece) {
      if ($piece == '.') {
        $piece[$i] = '';
      } else if ($piece == '..' && $i > 0) {
        $pieces[$i] = '';
        while ($i > 0 && !$pieces[$i-1]) {
          $i--;
        }
        $pieces[$i-1] = '';
      }
    }
    // strlen is used to remove empty strings, but keep values of 0 (zero)
    return ($starts_with_slash ? '/' : '') .
           implode('/', array_filter($pieces, 'strlen'));
  }

  /**
   * BELOW THIS ISN'T PART OF THE ZEND API. THEY ARE FOR THE STREAM WRAPPER.
   */

  /**
   * For the stream wrapper to stat a file. Same response format as stat().
   * Called from C++.
   */
  private static function stat($full_filename) {
    list($phar, $filename) = self::getPharAndFile($full_filename);
    if (!$phar->offsetExists($filename)) {
      $dir = self::opendir($full_filename);
      if (!$dir) {
        return false;
      }

      return array(
        'size' => 0,
        'atime' => 0,
        'mtime' => 0,
        'ctime' => 0,
        'mode' => POSIX_S_IFDIR,
      );
    }

    $info = $phar->offsetGet($filename);
    return array(
      'size' => $info->getSize(),
      'atime' => $info->getTimestamp(),
      'mtime' => $info->getTimestamp(),
      'ctime' => $info->getTimestamp(),
      'mode' => POSIX_S_IFREG,
    );
  }

  /**
   * Simulates opendir() and readdir() and rewinddir() using an array.
   * Returns any files that start with $prefix.
   * Called from C++.
   */
  private static function opendir($full_prefix) {
    list($phar, $prefix) = self::getPharAndFile($full_prefix);
    $prefix = rtrim($prefix, '/');

    $ret = array();
    foreach ($phar->archiveHandler->getEntriesMap()->keys() as $filename) {
      if (!$prefix) {
        if (strpos($filename, '/') === false) {
          $ret[$filename] = true;
        }
      } else {
        if (strpos($filename, $prefix) === 0) {
          $entry = substr($filename, strlen($prefix) + 1);
          if (strlen($entry) > 0) {
            if ($filename[strlen($prefix)] != '/') {
              continue;
            }
            $next_slash = strpos($entry, '/');
            if ($next_slash !== false) {
              $entry = substr($entry, 0, $next_slash);
            }
            $ret[$entry] = true;
          }
        }
      }
    }
    return array_keys($ret);
  }

  /**
   * Used by the stream wrapper to open phar:// files.
   * Called from C++.
   */
  private static function openPhar(string $full_filename): resource {
    list($phar, $filename) = self::getPharAndFile($full_filename);
    return $phar->getFileData($filename);
  }

  private function getFileData(string $filename): resource {
    if ($filename !== '') {
      $stream = $this->archiveHandler->getStream($filename);
    } else if ($stub = $this->getStub()) {
      $stream = fopen('php://temp', 'w+b');
      fwrite($stream, $stub);
      rewind($stream);
    } else {
      throw new PharException("No $filename in phar");
    }
    return $stream;
  }

  /**
   * Checks through a phar://path/to/file.phar/other/path.php and returns
   *
   *   array([Phar object for path/to/file.phar], 'other/path.php')
   *
   * or if the first piece is a valid alias, then returns
   *
   *   array([Phar object for alias], 'rest/of/path.php')
   */
  private static function getPharAndFile(string $filename_or_alias) {
    /**
     * TODO: This is a hack, during `include 'phar:///path/to.phar';`
     * `self::stat()` calls this method with `$filename_or_alias` that have
     * double `phar://phar://` prefix; I have no idea why, but we can fix it
     * here for now
     */
    if (strpos($filename_or_alias, 'phar://phar://') === 0) {
      $filename_or_alias = substr($filename_or_alias, 7);
    }
    if (strpos($filename_or_alias, 'phar://') !== 0) {
      throw new PharException("Not a phar: $filename_or_alias");
    }

    $pieces = explode('/', substr($filename_or_alias, 7));

    if (count($pieces) > 0 && isset(self::$aliases[$pieces[0]])) {
      $alias = array_shift($pieces);
      return array(
        self::$aliases[$alias],
        self::resolveDotDots($pieces)
      );
    }

    $filename = '';
    while ($pieces) {
      $filename .= '/'.array_shift($pieces);
      if (is_file($filename)) {

        if (!isset(self::$aliases[$filename])) {
          self::loadPhar($filename);
        }

        return array(
          self::$aliases[$filename],
          self::resolveDotDots($pieces)
        );
      }
    }

    throw new PharException("Not a phar: $filename_or_alias");
  }

  protected function getIteratorFromList(
    string $root,
    array<string, PharFileInfo> $list,
  ) {
    $tree = array();
    foreach ($list as $filename => $info) {
      $dir = dirname($filename);
      $current = &$tree;
      if ($dir !== '') {
        $path = $root;
        foreach (explode('/', $dir) as $part) {
          $path .= $part.'/';
          if (!isset($current[$path])) {
            $current[$path] = array();
          }
          $current = &$current[$path];
        }
      }
      $current[$root.$filename] = $info;
    }
    return new RecursiveArrayIterator(
      $tree,
      RecursiveArrayIterator::CHILD_ARRAYS_ONLY
    );
  }

  protected function getIterator() {
    if ($this->iterator !== null) {
      return $this->iterator;
    }
    $info = [];
    foreach ($this->archiveHandler->getEntriesMap() as $filename => $entry) {
      if ($entry->type === \__SystemLib\ArchiveEntryType::DIRECTORY) {
        continue;
      }
      $info[$filename] = new PharFileInfo(
        $this->iteratorRoot.$filename,
        $entry
      );
    }
    $this->iterator = $this->getIteratorFromList($this->iteratorRoot, $info);
    return $this->iterator;
  }

  public function key() {
    return $this->getIterator()->key();
  }

  public function current() {
    return $this->getIterator()->current();
  }

  public function next() {
    $this->getIterator()->next();
  }

  public function rewind() {
    $this->getIterator()->rewind();
  }

  public function valid() {
    return $this->getIterator()->valid();
  }

  public function hasChildren() {
    return $this->getIterator()->hasChildren();
  }

  public function getChildren() {
    return $this->getIterator()->getChildren();
  }

  protected static function assertWriteSupport() {
    if (ini_get('phar.readonly') != 0) {
      throw new UnexpectedValueException(
        'Cannot write out phar archive, phar is read-only'
      );
    }
  }
}
