/*
 * Global Presence - wraps calls to set and get presence for all accounts.
 *
 * Copyright (C) 2011 David Edmundson <kde@davidedmundson.co.uk>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifndef GLOBALPRESENCE_H
#define GLOBALPRESENCE_H

#include <QObject>
#include <QIcon>
#include <QQmlListProperty>

#include <TelepathyQt/AccountManager>
#include <TelepathyQt/AccountSet>
#include <TelepathyQt/Constants>

#include <QLoggingCategory>
// include the QDebug here so there doesn't have to
// be two debug includes in the files using qCDebug
Q_DECLARE_LOGGING_CATEGORY(KTP_LEGACY_PRESENCE)

#include "presence.h"

class QDBusServiceWatcher;
class QDBusPendingCallWatcher;

namespace KTpLegacy
{

class PresenceObject;

/** This class handles the presence between all enabled accounts
 * It shows the highest current available presence, indicates if any accounts are changing, and what they are changing to.
*/

class GlobalPresence : public QObject
{
    Q_OBJECT
    Q_ENUMS(ConnectionPresenceType)
    Q_PROPERTY(Tp::AccountManagerPtr accountManager READ accountManager)
    Q_PROPERTY(QQmlListProperty<KTpLegacy::PresenceObject> presencesList READ presencesList)

    Q_PROPERTY(QString presenceMessage READ currentPresenceMessage NOTIFY currentPresenceChanged)
    Q_PROPERTY(ConnectionPresenceType presenceType READ currentPresenceType NOTIFY currentPresenceChanged)
    Q_PROPERTY(QIcon currentPresenceIcon READ currentPresenceIcon NOTIFY currentPresenceChanged)
    Q_PROPERTY(QString currentPresenceIconName READ currentPresenceIconName NOTIFY currentPresenceChanged)
    Q_PROPERTY(KTpLegacy::Presence currentPresence READ currentPresence NOTIFY currentPresenceChanged)
    Q_PROPERTY(QString currentPresenceName READ currentPresenceName NOTIFY currentPresenceChanged);
    Q_PROPERTY(KTpLegacy::Presence requestedPresence READ requestedPresence WRITE setPresence NOTIFY requestedPresenceChanged)
    Q_PROPERTY(QString requestedPresenceName READ requestedPresenceName NOTIFY requestedPresenceChanged)
    Q_PROPERTY(bool isChangingPresence READ isChangingPresence NOTIFY connectionStatusChanged)

public:
    explicit GlobalPresence(QObject *parent = 0);

    enum ConnectionPresenceType
    {
        Offline = Tp::ConnectionPresenceTypeOffline,
        Available = Tp::ConnectionPresenceTypeAvailable,
        Away = Tp::ConnectionPresenceTypeAway,
        ExtendedAway = Tp::ConnectionPresenceTypeExtendedAway,
        Hidden = Tp::ConnectionPresenceTypeHidden,
        Busy = Tp::ConnectionPresenceTypeBusy,
        Unknown = Tp::ConnectionPresenceTypeUnknown,
        Error = Tp::ConnectionPresenceTypeError
    };
    Q_ENUMS(ConnectionPresenceType)

    /** Set the account manager to use
      * @param accountManager should be ready.
      */
    void setAccountManager(const Tp::AccountManagerPtr &accountManager);

    Tp::AccountManagerPtr accountManager() const;

    /** Returns connecting if any account is connecting, else connected if at least one account is connected, disconnected otherwise*/
    Tp::ConnectionStatus connectionStatus() const;

    /** The most online presence of any account*/
    Presence currentPresence() const;
    QString currentPresenceMessage() const;
    QIcon currentPresenceIcon() const;
    QString currentPresenceIconName() const;
    ConnectionPresenceType currentPresenceType() const;
    QString currentPresenceName() const;

    /** The most online presence requested for any account if any of the accounts are changing state.
      otherwise returns current presence*/
    Presence requestedPresence() const;

    QString requestedPresenceName() const;

    /** Returns true if any account is changing state (i.e connecting*/
    bool isChangingPresence() const;

    /** Returns true if there is any enabled account */
    bool hasEnabledAccounts() const;

    Tp::AccountSetPtr onlineAccounts() const;

    Q_INVOKABLE void toggleContactList() const;

    QQmlListProperty<KTpLegacy::PresenceObject> presencesList();

    Q_INVOKABLE void setPresenceByIndex(int index);


Q_SIGNALS:
    void requestedPresenceChanged(const KTpLegacy::Presence &customPresence);
    void currentPresenceChanged(const KTpLegacy::Presence &presence);
    void changingPresence(bool isChanging);
    void connectionStatusChanged(Tp::ConnectionStatus);
    void accountManagerReady();

public Q_SLOTS:
    /** Set all enabled accounts to the specified presence*/
    void setPresence(const KTpLegacy::Presence &presence);
    void setPresence(ConnectionPresenceType p, QString message);

    /**Saves the current presence to memory*/
    void saveCurrentPresence();
    /**Restores the saved presence from memory */
    void restoreSavedPresence();

private Q_SLOTS:
    void onCurrentPresenceChanged();
    void onRequestedPresenceChanged();
    void onChangingPresence();
    void onConnectionStatusChanged();

    void onAccountAdded(const Tp::AccountPtr &account);
    void onAccountManagerReady(Tp::PendingOperation* op);
    void serviceNameFetchFinished(QDBusPendingCallWatcher *callWatcher);

private:
    Tp::AccountManagerPtr m_accountManager;

    Tp::AccountSetPtr m_enabledAccounts;
    Tp::AccountSetPtr m_onlineAccounts;

    /**Saved presence for later restoration (for example after returning from auto-away) */
    KTpLegacy::Presence m_savedPresence;
    /** A cache of the last sent requested presence, to avoid resignalling*/
    KTpLegacy::Presence m_requestedPresence;
    /** A cache of the last sent presence*/
    KTpLegacy::Presence m_currentPresence;

    Tp::ConnectionStatus m_connectionStatus;
    QList<KTpLegacy::PresenceObject*> m_presencesList;
    QDBusServiceWatcher  *m_contactListWatcher;
    bool m_changingPresence;
    bool m_contactListRunning;
};

}

#endif // GLOBALPRESENCE_H
