/*
 * Contributed by Stephane Eranian <eranian@gmail.com>
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies
 * of the Software, and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
 * INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
 * CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE
 * OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * This file is part of libpfm, a performance monitoring support library for
 * applications on Linux.
 *
 * PMU: adl_grt (Alderlake GoldenCove E-Core)
 * Based on Intel JSON event table version   : 1.24
 * Based on Intel JSON event table published : 12/04/2023
 */

static const intel_x86_umask_t adl_grt_baclears[]={
  { .uname   = "ANY",
    .udesc   = "Counts the total number of BACLEARS due to all branch types including conditional and unconditional jumps, returns, and indirect branches",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_DFL,
  },
};

static const intel_x86_umask_t adl_grt_br_inst_retired[]={
  { .uname   = "ALL_BRANCHES",
    .udesc   = "Counts the total number of branch instructions retired for all branch types",
    .ucode   = 0x0000ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS | INTEL_X86_DFL,
  },
  { .uname   = "CALL",
    .udesc   = "This event is deprecated. Refer to new event BR_INST_RETIRED.NEAR_CALL",
    .uequiv  = "NEAR_CALL",
    .ucode   = 0xf900ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "COND",
    .udesc   = "Counts the number of retired JCC (Jump on Conditional Code) branch instructions retired, includes both taken and not taken branches",
    .ucode   = 0x7e00ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "COND_TAKEN",
    .udesc   = "Counts the number of taken JCC (Jump on Conditional Code) branch instructions retired",
    .ucode   = 0xfe00ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "FAR_BRANCH",
    .udesc   = "Counts the number of far branch instructions retired, includes far jump, far call and return, and interrupt call and return",
    .ucode   = 0xbf00ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "INDIRECT",
    .udesc   = "Counts the number of near indirect JMP and near indirect CALL branch instructions retired",
    .ucode   = 0xeb00ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "INDIRECT_CALL",
    .udesc   = "Counts the number of near indirect CALL branch instructions retired",
    .ucode   = 0xfb00ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "IND_CALL",
    .udesc   = "This event is deprecated. Refer to new event BR_INST_RETIRED.INDIRECT_CALL",
    .uequiv  = "INDIRECT_CALL",
    .ucode   = 0xfb00ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "JCC",
    .udesc   = "This event is deprecated. Refer to new event BR_INST_RETIRED.COND",
    .uequiv  = "COND",
    .ucode   = 0x7e00ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "NEAR_CALL",
    .udesc   = "Counts the number of near CALL branch instructions retired",
    .ucode   = 0xf900ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "NEAR_RETURN",
    .udesc   = "Counts the number of near RET branch instructions retired",
    .ucode   = 0xf700ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "NEAR_TAKEN",
    .udesc   = "Counts the number of near taken branch instructions retired",
    .ucode   = 0xc000ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "NON_RETURN_IND",
    .udesc   = "This event is deprecated. Refer to new event BR_INST_RETIRED.INDIRECT",
    .uequiv  = "INDIRECT",
    .ucode   = 0xeb00ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "REL_CALL",
    .udesc   = "Counts the number of near relative CALL branch instructions retired",
    .ucode   = 0xfd00ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "RETURN",
    .udesc   = "This event is deprecated. Refer to new event BR_INST_RETIRED.NEAR_RETURN",
    .uequiv  = "NEAR_RETURN",
    .ucode   = 0xf700ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "TAKEN_JCC",
    .udesc   = "This event is deprecated. Refer to new event BR_INST_RETIRED.COND_TAKEN",
    .uequiv  = "COND_TAKEN",
    .ucode   = 0xfe00ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
};

static const intel_x86_umask_t adl_grt_br_misp_retired[]={
  { .uname   = "ALL_BRANCHES",
    .udesc   = "Counts the total number of mispredicted branch instructions retired for all branch types",
    .ucode   = 0x0000ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS | INTEL_X86_DFL,
  },
  { .uname   = "COND",
    .udesc   = "Counts the number of mispredicted JCC (Jump on Conditional Code) branch instructions retired",
    .ucode   = 0x7e00ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "COND_TAKEN",
    .udesc   = "Counts the number of mispredicted taken JCC (Jump on Conditional Code) branch instructions retired",
    .ucode   = 0xfe00ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "INDIRECT",
    .udesc   = "Counts the number of mispredicted near indirect JMP and near indirect CALL branch instructions retired",
    .ucode   = 0xeb00ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "INDIRECT_CALL",
    .udesc   = "Counts the number of mispredicted near indirect CALL branch instructions retired",
    .ucode   = 0xfb00ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "IND_CALL",
    .udesc   = "This event is deprecated. Refer to new event BR_MISP_RETIRED.INDIRECT_CALL",
    .uequiv  = "INDIRECT_CALL",
    .ucode   = 0xfb00ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "JCC",
    .udesc   = "This event is deprecated. Refer to new event BR_MISP_RETIRED.COND",
    .uequiv  = "COND",
    .ucode   = 0x7e00ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "NEAR_TAKEN",
    .udesc   = "Counts the number of mispredicted near taken branch instructions retired",
    .ucode   = 0x8000ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "NON_RETURN_IND",
    .udesc   = "This event is deprecated. Refer to new event BR_MISP_RETIRED.INDIRECT",
    .uequiv  = "INDIRECT",
    .ucode   = 0xeb00ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "RETURN",
    .udesc   = "Counts the number of mispredicted near RET branch instructions retired",
    .ucode   = 0xf700ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "TAKEN_JCC",
    .udesc   = "This event is deprecated. Refer to new event BR_MISP_RETIRED.COND_TAKEN",
    .uequiv  = "COND_TAKEN",
    .ucode   = 0xfe00ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
};

static const intel_x86_umask_t adl_grt_cpu_clk_unhalted[]={
  { .uname   = "CORE",
    .udesc   = "Counts the number of unhalted core clock cycles. (Fixed event)",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_CODE_OVERRIDE,
  },
  { .uname   = "CORE_P",
    .udesc   = "Counts the number of unhalted core clock cycles",
    .ucode   = 0x0000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "REF_TSC",
    .udesc   = "Counts the number of unhalted reference clock cycles at TSC frequency. (Fixed event)",
    .ucode   = 0x0300ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_CODE_OVERRIDE,
  },
  { .uname   = "REF_TSC_P",
    .udesc   = "Counts the number of unhalted reference clock cycles at TSC frequency",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "THREAD",
    .udesc   = "Counts the number of unhalted core clock cycles. (Fixed event)",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_CODE_OVERRIDE,
  },
  { .uname   = "THREAD_P",
    .udesc   = "Counts the number of unhalted core clock cycles",
    .ucode   = 0x0000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t adl_grt_dtlb_load_misses[]={
  { .uname   = "WALK_COMPLETED",
    .udesc   = "Counts the number of page walks completed due to load DTLB misses to any page size",
    .ucode   = 0x0e00ull,
    .uflags  = INTEL_X86_DFL,
  },
};

static const intel_x86_umask_t adl_grt_dtlb_store_misses[]={
  { .uname   = "WALK_COMPLETED",
    .udesc   = "Counts the number of page walks completed due to store DTLB misses to any page size",
    .ucode   = 0x0e00ull,
    .uflags  = INTEL_X86_DFL,
  },
};

static const intel_x86_umask_t adl_grt_icache[]={
  { .uname   = "ACCESSES",
    .udesc   = "Counts the number of requests to the instruction cache for one or more bytes of a cache line",
    .ucode   = 0x0300ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "MISSES",
    .udesc   = "Counts the number of instruction cache misses",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t adl_grt_inst_retired[]={
  { .uname   = "ANY",
    .udesc   = "Counts the total number of instructions retired. (Fixed event)",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS | INTEL_X86_CODE_OVERRIDE,
  },
  { .uname   = "ANY_P",
    .udesc   = "Counts the total number of instructions retired",
    .ucode   = 0x0000ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS | INTEL_X86_DFL,
  },
};

static const intel_x86_umask_t adl_grt_itlb_misses[]={
  { .uname   = "MISS_CAUSED_WALK",
    .udesc   = "Counts the number of page walks initiated by a instruction fetch that missed the first and second level TLBs",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "PDE_CACHE_MISS",
    .udesc   = "Counts the number of page walks due to an instruction fetch that miss the PDE (Page Directory Entry) cache",
    .ucode   = 0x8000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "WALK_COMPLETED",
    .udesc   = "Counts the number of page walks completed due to instruction fetch misses to any page size",
    .ucode   = 0x0e00ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t adl_grt_lbr_inserts[]={
  { .uname   = "ANY",
    .udesc   = "This event is deprecated. [This event is alias to MISC_RETIRED.LBR_INSERTS]",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_PEBS | INTEL_X86_DFL,
  },
};

static const intel_x86_umask_t adl_grt_ld_blocks[]={
  { .uname   = "4K_ALIAS",
    .udesc   = "This event is deprecated. Refer to new event LD_BLOCKS.ADDRESS_ALIAS",
    .ucode   = 0x0400ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "ADDRESS_ALIAS",
    .udesc   = "Counts the number of retired loads that are blocked because it initially appears to be store forward blocked, but subsequently is shown not to be blocked based on 4K alias check",
    .ucode   = 0x0400ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "DATA_UNKNOWN",
    .udesc   = "Counts the number of retired loads that are blocked because its address exactly matches an older store whose data is not ready",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
};

static const intel_x86_umask_t adl_grt_ld_head[]={
  { .uname   = "ANY_AT_RET",
    .udesc   = "Counts the number of cycles that the head (oldest load) of the load buffer is stalled due to any number of reasons, including an L1 miss, WCB full, pagewalk, store address block or store data block, on a load that retires",
    .ucode   = 0xff00ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_DFL,
  },
  { .uname   = "DTLB_MISS_AT_RET",
    .udesc   = "Counts the number of cycles that the head (oldest load) of the load buffer and retirement are both stalled due to a DTLB miss",
    .ucode   = 0x9000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "L1_BOUND_AT_RET",
    .udesc   = "Counts the number of cycles that the head (oldest load) of the load buffer is stalled due to a core bound stall including a store address match, a DTLB miss or a page walk that detains the load from retiring",
    .ucode   = 0xf400ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "L1_MISS_AT_RET",
    .udesc   = "Counts the number of cycles that the head (oldest load) of the load buffer and retirement are both stalled due to a DL1 miss",
    .ucode   = 0x8100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "OTHER_AT_RET",
    .udesc   = "Counts the number of cycles that the head (oldest load) of the load buffer and retirement are both stalled due to other block cases",
    .ucode   = 0xc000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "PGWALK_AT_RET",
    .udesc   = "Counts the number of cycles that the head (oldest load) of the load buffer and retirement are both stalled due to a pagewalk",
    .ucode   = 0xa000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "ST_ADDR_AT_RET",
    .udesc   = "Counts the number of cycles that the head (oldest load) of the load buffer and retirement are both stalled due to a store address match",
    .ucode   = 0x8400ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t adl_grt_longest_lat_cache[]={
  { .uname   = "MISS",
    .udesc   = "Counts the number of cacheable memory requests that miss in the LLC. Counts on a per core basis",
    .ucode   = 0x4100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "REFERENCE",
    .udesc   = "Counts the number of cacheable memory requests that access the LLC. Counts on a per core basis",
    .ucode   = 0x4f00ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t adl_grt_machine_clears[]={
  { .uname   = "DISAMBIGUATION",
    .udesc   = "Counts the number of machine clears due to memory ordering in which an internal load passes an older store within the same CPU",
    .ucode   = 0x0800ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "FP_ASSIST",
    .udesc   = "Counts the number of floating point operations retired that required microcode assist",
    .ucode   = 0x0400ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "MEMORY_ORDERING",
    .udesc   = "Counts the number of machine clears due to memory ordering caused by a snoop from an external agent. Does not count internally generated machine clears such as those due to memory disambiguation",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "MRN_NUKE",
    .udesc   = "Counts the number of machines clears due to memory renaming",
    .ucode   = 0x8000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "PAGE_FAULT",
    .udesc   = "Counts the number of machine clears due to a page fault.  Counts both I-Side and D-Side (Loads/Stores) page faults.  A page fault occurs when either the page is not present, or an access violation occurs",
    .ucode   = 0x2000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "SLOW",
    .udesc   = "Counts the number of machine clears that flush the pipeline and restart the machine with the use of microcode due to SMC, MEMORY_ORDERING, FP_ASSISTS, PAGE_FAULT, DISAMBIGUATION, and FPC_VIRTUAL_TRAP",
    .ucode   = 0x6f00ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "SMC",
    .udesc   = "Counts the number of machine clears due to program modifying data (self modifying code) within 1K of a recently fetched code page",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t adl_grt_mem_bound_stalls[]={
  { .uname   = "IFETCH",
    .udesc   = "Counts the number of cycles the core is stalled due to an instruction cache or TLB miss which hit in the L2, LLC, DRAM or MMIO (Non-DRAM)",
    .ucode   = 0x3800ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "IFETCH_DRAM_HIT",
    .udesc   = "Counts the number of cycles the core is stalled due to an instruction cache or TLB miss which hit in DRAM or MMIO (Non-DRAM)",
    .ucode   = 0x2000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "IFETCH_L2_HIT",
    .udesc   = "Counts the number of cycles the core is stalled due to an instruction cache or TLB miss which hit in the L2 cache",
    .ucode   = 0x0800ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "IFETCH_LLC_HIT",
    .udesc   = "Counts the number of cycles the core is stalled due to an instruction cache or TLB miss which hit in the LLC or other core with HITE/F/M",
    .ucode   = 0x1000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "LOAD",
    .udesc   = "Counts the number of cycles the core is stalled due to a demand load miss which hit in the L2, LLC, DRAM or MMIO (Non-DRAM)",
    .ucode   = 0x0700ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "LOAD_DRAM_HIT",
    .udesc   = "Counts the number of cycles the core is stalled due to a demand load miss which hit in DRAM or MMIO (Non-DRAM)",
    .ucode   = 0x0400ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "LOAD_L2_HIT",
    .udesc   = "Counts the number of cycles the core is stalled due to a demand load which hit in the L2 cache",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "LOAD_LLC_HIT",
    .udesc   = "Counts the number of cycles the core is stalled due to a demand load which hit in the LLC or other core with HITE/F/M",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t adl_grt_mem_load_uops_retired[]={
  { .uname   = "DRAM_HIT",
    .udesc   = "Counts the number of load uops retired that hit in DRAM",
    .ucode   = 0x8000ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "L2_HIT",
    .udesc   = "Counts the number of load uops retired that hit in the L2 cache",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "L3_HIT",
    .udesc   = "Counts the number of load uops retired that hit in the L3 cache",
    .ucode   = 0x0400ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
};

static const intel_x86_umask_t adl_grt_mem_scheduler_block[]={
  { .uname   = "ALL",
    .udesc   = "load buffer, store buffer or RSV full",
    .ucode   = 0x0700ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_DFL,
  },
  { .uname   = "LD_BUF",
    .udesc   = "Counts the number of cycles that uops are blocked due to a load buffer full condition",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "RSV",
    .udesc   = "Counts the number of cycles that uops are blocked due to an RSV full condition",
    .ucode   = 0x0400ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "ST_BUF",
    .udesc   = "Counts the number of cycles that uops are blocked due to a store buffer full condition",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t adl_grt_mem_uops_retired[]={
  { .uname   = "ALL_LOADS",
    .udesc   = "Counts the number of load uops retired",
    .ucode   = 0x8100ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "ALL_STORES",
    .udesc   = "Counts the number of store uops retired",
    .ucode   = 0x8200ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname  = "LOAD_LATENCY",
    .udesc  = "Memory load instructions retired above programmed clocks, minimum threshold value is 3 (Precise Event and ldlat required)",
    .ucode  = 0x500ull,
    .uflags = INTEL_X86_NCOMBO | INTEL_X86_PEBS | INTEL_X86_LDLAT | INTEL_X86_DFL,
  },

  { .uname   = "LOAD_LATENCY_GT_128",
    .udesc   = "Counts the number of tagged loads with an instruction latency that exceeds or equals the threshold of 128 cycles as defined in MEC_CR_PEBS_LD_LAT_THRESHOLD (3F6H). Only counts with PEBS enabled",
    .ucode   = 0x8000ull,
    .uequiv  = "LOAD_LATENCY:ldlat=128",
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS | INTEL_X86_LDLAT,
  },
  { .uname   = "LOAD_LATENCY_GT_16",
    .udesc   = "Counts the number of tagged loads with an instruction latency that exceeds or equals the threshold of 16 cycles as defined in MEC_CR_PEBS_LD_LAT_THRESHOLD (3F6H). Only counts with PEBS enabled",
    .ucode   = 0x1000ull,
    .uequiv  = "LOAD_LATENCY:ldlat=16",
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS | INTEL_X86_LDLAT,
  },
  { .uname   = "LOAD_LATENCY_GT_256",
    .udesc   = "Counts the number of tagged loads with an instruction latency that exceeds or equals the threshold of 256 cycles as defined in MEC_CR_PEBS_LD_LAT_THRESHOLD (3F6H). Only counts with PEBS enabled",
    .ucode   = 0x10000ull,
    .uequiv  = "LOAD_LATENCY:ldlat=256",
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS | INTEL_X86_LDLAT,
  },
  { .uname   = "LOAD_LATENCY_GT_32",
    .udesc   = "Counts the number of tagged loads with an instruction latency that exceeds or equals the threshold of 32 cycles as defined in MEC_CR_PEBS_LD_LAT_THRESHOLD (3F6H). Only counts with PEBS enabled",
    .ucode   = 0x2000ull,
    .uequiv  = "LOAD_LATENCY:ldlat=32",
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "LOAD_LATENCY_GT_4",
    .udesc   = "Counts the number of tagged loads with an instruction latency that exceeds or equals the threshold of 4 cycles as defined in MEC_CR_PEBS_LD_LAT_THRESHOLD (3F6H). Only counts with PEBS enabled",
    .ucode   = 0x0400ull,
    .uequiv  = "LOAD_LATENCY:ldlat=4",
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS | INTEL_X86_LDLAT,
  },
  { .uname   = "LOAD_LATENCY_GT_512",
    .udesc   = "Counts the number of tagged loads with an instruction latency that exceeds or equals the threshold of 512 cycles as defined in MEC_CR_PEBS_LD_LAT_THRESHOLD (3F6H). Only counts with PEBS enabled",
    .ucode   = 0x20000ull,
    .uequiv  = "LOAD_LATENCY:ldlat=512",
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS | INTEL_X86_LDLAT,
  },
  { .uname   = "LOAD_LATENCY_GT_64",
    .udesc   = "Counts the number of tagged loads with an instruction latency that exceeds or equals the threshold of 64 cycles as defined in MEC_CR_PEBS_LD_LAT_THRESHOLD (3F6H). Only counts with PEBS enabled",
    .ucode   = 0x4000ull,
    .uequiv  = "LOAD_LATENCY:ldlat=64",
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "LOAD_LATENCY_GT_8",
    .udesc   = "Counts the number of tagged loads with an instruction latency that exceeds or equals the threshold of 8 cycles as defined in MEC_CR_PEBS_LD_LAT_THRESHOLD (3F6H). Only counts with PEBS enabled",
    .ucode   = 0x0800ull,
    .uequiv  = "LOAD_LATENCY:ldlat=8",
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS | INTEL_X86_LDLAT,
  },
  { .uname   = "SPLIT_LOADS",
    .udesc   = "Counts the number of retired split load uops",
    .ucode   = 0x4100ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "STORE_LATENCY",
    .udesc   = "Counts the number of stores uops retired. Counts with or without PEBS enabled",
    .ucode   = 0x0600ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
};

static const intel_x86_umask_t adl_grt_misc_retired[]={
  { .uname   = "LBR_INSERTS",
    .udesc   = "Counts the number of LBR entries recorded. Requires LBRs to be enabled in IA32_LBR_CTL. [This event is alias to LBR_INSERTS.ANY]",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_PEBS | INTEL_X86_DFL,
  },
};

static const intel_x86_umask_t adl_grt_ocr[]={
  { .uname   = "COREWB_M_ANY_RESPONSE",
    .udesc   = "Counts modified writebacks from L1 cache and L2 cache that have any type of response",
    .ucode   = 0x1000800ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "DEMAND_DATA_RD_ANY_RESPONSE",
    .udesc   = "Counts demand data reads that have any type of response",
    .ucode   = 0x1000100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "DEMAND_DATA_RD_L3_HIT",
    .udesc   = "Counts demand data reads that were supplied by the L3 cache",
    .ucode   = 0x3f803c000100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "DEMAND_DATA_RD_L3_HIT_SNOOP_HITM",
    .udesc   = "Counts demand data reads that were supplied by the L3 cache where a snoop was sent, the snoop hit, and modified data was forwarded",
    .ucode   = 0x10003c000100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "DEMAND_DATA_RD_L3_HIT_SNOOP_HIT_NO_FWD",
    .udesc   = "Counts demand data reads that were supplied by the L3 cache where a snoop was sent, the snoop hit, but no data was forwarded",
    .ucode   = 0x4003c000100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "DEMAND_DATA_RD_L3_HIT_SNOOP_HIT_WITH_FWD",
    .udesc   = "Counts demand data reads that were supplied by the L3 cache where a snoop was sent, the snoop hit, and non-modified data was forwarded",
    .ucode   = 0x8003c000100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "DEMAND_DATA_RD_L3_MISS",
    .udesc   = "Counts demand data reads that were not supplied by the L3 cache",
    .ucode   = 0x3f8440000100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "DEMAND_DATA_RD_L3_MISS_LOCAL",
    .udesc   = "Counts demand data reads that were not supplied by the L3 cache. [L3_MISS_LOCAL is alias to L3_MISS]",
    .ucode   = 0x3f8440000100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "DEMAND_RFO_ANY_RESPONSE",
    .udesc   = "Counts demand reads for ownership (RFO) and software prefetches for exclusive ownership (PREFETCHW) that have any type of response",
    .ucode   = 0x1000200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "DEMAND_RFO_L3_HIT",
    .udesc   = "Counts demand reads for ownership (RFO) and software prefetches for exclusive ownership (PREFETCHW) that were supplied by the L3 cache",
    .ucode   = 0x3f803c000200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "DEMAND_RFO_L3_HIT_SNOOP_HITM",
    .udesc   = "Counts demand reads for ownership (RFO) and software prefetches for exclusive ownership (PREFETCHW) that were supplied by the L3 cache where a snoop was sent, the snoop hit, and modified data was forwarded",
    .ucode   = 0x10003c000200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "DEMAND_RFO_L3_MISS",
    .udesc   = "Counts demand reads for ownership (RFO) and software prefetches for exclusive ownership (PREFETCHW) that were not supplied by the L3 cache",
    .ucode   = 0x3f8440000200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "DEMAND_RFO_L3_MISS_LOCAL",
    .udesc   = "Counts demand reads for ownership (RFO) and software prefetches for exclusive ownership (PREFETCHW) that were not supplied by the L3 cache. [L3_MISS_LOCAL is alias to L3_MISS]",
    .ucode   = 0x3f8440000200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "STREAMING_WR_ANY_RESPONSE",
    .udesc   = "Counts streaming stores that have any type of response",
    .ucode   = 0x1080000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t adl_grt_serialization[]={
  { .uname   = "NON_C01_MS_SCB",
    .udesc   = "Counts the number of issue slots not consumed by the backend due to a micro-sequencer (MS) scoreboard, which stalls the front-end from issuing from the UROM until a specified older uop retires",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_DFL,
  },
};

static const intel_x86_umask_t adl_grt_topdown_bad_speculation[]={
  { .uname   = "ALL",
    .udesc   = "Counts the total number of issue slots that were not consumed by the backend because allocation is stalled due to a mispredicted jump or a machine clear",
    .ucode   = 0x0000ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_DFL,
  },
  { .uname   = "FASTNUKE",
    .udesc   = "Counts the number of issue slots every cycle that were not consumed by the backend due to fast nukes such as memory ordering and memory disambiguation machine clears",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "MACHINE_CLEARS",
    .udesc   = "Counts the total number of issue slots that were not consumed by the backend because allocation is stalled due to a machine clear (nuke) of any kind including memory ordering and memory disambiguation",
    .ucode   = 0x0300ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "MISPREDICT",
    .udesc   = "Counts the number of issue slots every cycle that were not consumed by the backend due to branch mispredicts",
    .ucode   = 0x0400ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "NUKE",
    .udesc   = "Counts the number of issue slots every cycle that were not consumed by the backend due to a machine clear (nuke)",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t adl_grt_topdown_be_bound[]={
  { .uname   = "ALL",
    .udesc   = "Counts the total number of issue slots every cycle that were not consumed by the backend due to backend stalls",
    .ucode   = 0x0000ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_DFL,
  },
  { .uname   = "ALLOC_RESTRICTIONS",
    .udesc   = "Counts the number of issue slots every cycle that were not consumed by the backend due to certain allocation restrictions",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "MEM_SCHEDULER",
    .udesc   = "Counts the number of issue slots every cycle that were not consumed by the backend due to memory reservation stalls in which a scheduler is not able to accept uops",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "NON_MEM_SCHEDULER",
    .udesc   = "Counts the number of issue slots every cycle that were not consumed by the backend due to IEC or FPC RAT stalls, which can be due to FIQ or IEC reservation stalls in which the integer, floating point or SIMD scheduler is not able to accept uops",
    .ucode   = 0x0800ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "REGISTER",
    .udesc   = "Counts the number of issue slots every cycle that were not consumed by the backend due to the physical register file unable to accept an entry (marble stalls)",
    .ucode   = 0x2000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "REORDER_BUFFER",
    .udesc   = "Counts the number of issue slots every cycle that were not consumed by the backend due to the reorder buffer being full (ROB stalls)",
    .ucode   = 0x4000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "SERIALIZATION",
    .udesc   = "Counts the number of issue slots every cycle that were not consumed by the backend due to scoreboards from the instruction queue (IQ), jump execution unit (JEU), or microcode sequencer (MS)",
    .ucode   = 0x1000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t adl_grt_topdown_fe_bound[]={
  { .uname   = "ALL",
    .udesc   = "Counts the total number of issue slots every cycle that were not consumed by the backend due to frontend stalls",
    .ucode   = 0x0000ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_DFL,
  },
  { .uname   = "BRANCH_DETECT",
    .udesc   = "Counts the number of issue slots every cycle that were not delivered by the frontend due to BACLEARS",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "BRANCH_RESTEER",
    .udesc   = "Counts the number of issue slots every cycle that were not delivered by the frontend due to BTCLEARS",
    .ucode   = 0x4000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "CISC",
    .udesc   = "Counts the number of issue slots every cycle that were not delivered by the frontend due to the microcode sequencer (MS)",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "DECODE",
    .udesc   = "Counts the number of issue slots every cycle that were not delivered by the frontend due to decode stalls",
    .ucode   = 0x0800ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "FRONTEND_BANDWIDTH",
    .udesc   = "Counts the number of issue slots every cycle that were not delivered by the frontend due to frontend bandwidth restrictions due to decode, predecode, cisc, and other limitations",
    .ucode   = 0x8d00ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "FRONTEND_LATENCY",
    .udesc   = "Counts the number of issue slots every cycle that were not delivered by the frontend due to a latency related stalls including BACLEARs, BTCLEARs, ITLB misses, and ICache misses",
    .ucode   = 0x7200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "ICACHE",
    .udesc   = "Counts the number of issue slots every cycle that were not delivered by the frontend due to instruction cache misses",
    .ucode   = 0x2000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "ITLB",
    .udesc   = "Counts the number of issue slots every cycle that were not delivered by the frontend due to ITLB misses",
    .ucode   = 0x1000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "OTHER",
    .udesc   = "Counts the number of issue slots every cycle that were not delivered by the frontend due to other common frontend stalls not categorized",
    .ucode   = 0x8000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "PREDECODE",
    .udesc   = "Counts the number of issue slots every cycle that were not delivered by the frontend due to wrong predecodes",
    .ucode   = 0x0400ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t adl_grt_topdown_retiring[]={
  { .uname   = "ALL",
    .udesc   = "Counts the total number of consumed retirement slots",
    .ucode   = 0x0000ull,
    .uflags  = INTEL_X86_PEBS | INTEL_X86_DFL,
  },
};

static const intel_x86_umask_t adl_grt_uops_retired[]={
  { .uname   = "ALL",
    .udesc   = "Counts the total number of uops retired",
    .ucode   = 0x0000ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS | INTEL_X86_DFL,
  },
  { .uname   = "FPDIV",
    .udesc   = "Counts the number of floating point divide uops retired (x87 and SSE, including x87 sqrt)",
    .ucode   = 0x0800ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "IDIV",
    .udesc   = "Counts the number of integer divide uops retired",
    .ucode   = 0x1000ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "MS",
    .udesc   = "Counts the number of uops that are from complex flows issued by the micro-sequencer (MS)",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "X87",
    .udesc   = "Counts the number of x87 uops retired, includes those in MS flows",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
};

static const intel_x86_entry_t intel_adl_grt_pe[]={
  { .name   = "BACLEARS",
    .desc   = "Counts the total number of BACLEARS due to all branch types including conditional and unconditional jumps, returns, and indirect branches",
    .code   = 0x00e6,
    .modmsk = INTEL_V2_ATTRS,
    .cntmsk = 0x3full,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(adl_grt_baclears),
    .umasks = adl_grt_baclears,
  },
  { .name   = "BR_INST_RETIRED",
    .desc   = "Counts the total number of branch instructions retired for all branch types",
    .code   = 0x00c4,
    .modmsk = INTEL_V2_ATTRS,
    .cntmsk = 0x3full,
    .ngrp   = 1,
    .flags  = INTEL_X86_PEBS,
    .numasks= LIBPFM_ARRAY_SIZE(adl_grt_br_inst_retired),
    .umasks = adl_grt_br_inst_retired,
  },
  { .name   = "BR_MISP_RETIRED",
    .desc   = "Counts the total number of mispredicted branch instructions retired for all branch types",
    .code   = 0x00c5,
    .modmsk = INTEL_V2_ATTRS,
    .cntmsk = 0x3full,
    .ngrp   = 1,
    .flags  = INTEL_X86_PEBS,
    .numasks= LIBPFM_ARRAY_SIZE(adl_grt_br_misp_retired),
    .umasks = adl_grt_br_misp_retired,
  },
  { .name   = "CPU_CLK_UNHALTED",
    .desc   = "Counts the number of unhalted core clock cycles",
    .code   = 0x003c,
    .modmsk = INTEL_V2_ATTRS,
    .cntmsk = 0x1ull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(adl_grt_cpu_clk_unhalted),
    .umasks = adl_grt_cpu_clk_unhalted,
  },
  { .name   = "DTLB_LOAD_MISSES",
    .desc   = "Counts the number of page walks completed due to load DTLB misses to any page size",
    .code   = 0x0008,
    .modmsk = INTEL_V2_ATTRS,
    .cntmsk = 0x3full,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(adl_grt_dtlb_load_misses),
    .umasks = adl_grt_dtlb_load_misses,
  },
  { .name   = "DTLB_STORE_MISSES",
    .desc   = "Counts the number of page walks completed due to store DTLB misses to any page size",
    .code   = 0x0049,
    .modmsk = INTEL_V2_ATTRS,
    .cntmsk = 0x3full,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(adl_grt_dtlb_store_misses),
    .umasks = adl_grt_dtlb_store_misses,
  },
  { .name   = "ICACHE",
    .desc   = "Counts the number of instruction cache misses",
    .code   = 0x0080,
    .modmsk = INTEL_V2_ATTRS,
    .cntmsk = 0x3full,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(adl_grt_icache),
    .umasks = adl_grt_icache,
  },
  { .name   = "INST_RETIRED",
    .desc   = "Counts the total number of instructions retired. (Fixed event)",
    .code   = 0x00c0,
    .modmsk = INTEL_V2_ATTRS,
    .cntmsk = 0x1ull,
    .ngrp   = 1,
    .flags  = INTEL_X86_PEBS,
    .numasks= LIBPFM_ARRAY_SIZE(adl_grt_inst_retired),
    .umasks = adl_grt_inst_retired,
  },
  { .name   = "ITLB_MISSES",
    .desc   = "Counts the number of page walks initiated by a instruction fetch that missed the first and second level TLBs",
    .code   = 0x0085,
    .modmsk = INTEL_V2_ATTRS,
    .cntmsk = 0x3full,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(adl_grt_itlb_misses),
    .umasks = adl_grt_itlb_misses,
  },
  { .name   = "LBR_INSERTS",
    .desc   = "This event is deprecated. [This event is alias to MISC_RETIRED.LBR_INSERTS]",
    .code   = 0x00e4,
    .modmsk = INTEL_V2_ATTRS,
    .cntmsk = 0x3full,
    .ngrp   = 1,
    .flags  = INTEL_X86_PEBS | INTEL_X86_DEPRECATED,
    .numasks= LIBPFM_ARRAY_SIZE(adl_grt_lbr_inserts),
    .umasks = adl_grt_lbr_inserts,
  },
  { .name   = "LD_BLOCKS",
    .desc   = "Counts the number of retired loads that are blocked because its address exactly matches an older store whose data is not ready",
    .code   = 0x0003,
    .modmsk = INTEL_V2_ATTRS,
    .cntmsk = 0x3full,
    .ngrp   = 1,
    .flags  = INTEL_X86_PEBS,
    .numasks= LIBPFM_ARRAY_SIZE(adl_grt_ld_blocks),
    .umasks = adl_grt_ld_blocks,
  },
  { .name   = "LD_HEAD",
    .desc   = "Counts the number of cycles that the head (oldest load) of the load buffer and retirement are both stalled due to a DL1 miss",
    .code   = 0x0005,
    .modmsk = INTEL_V2_ATTRS,
    .cntmsk = 0x3full,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(adl_grt_ld_head),
    .umasks = adl_grt_ld_head,
  },
  { .name   = "LONGEST_LAT_CACHE",
    .desc   = "Counts the number of cacheable memory requests that miss in the LLC. Counts on a per core basis",
    .code   = 0x002e,
    .modmsk = INTEL_V2_ATTRS,
    .cntmsk = 0x3full,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(adl_grt_longest_lat_cache),
    .umasks = adl_grt_longest_lat_cache,
  },
  { .name   = "MACHINE_CLEARS",
    .desc   = "Counts the number of machine clears due to program modifying data (self modifying code) within 1K of a recently fetched code page",
    .code   = 0x00c3,
    .modmsk = INTEL_V2_ATTRS,
    .cntmsk = 0x3full,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(adl_grt_machine_clears),
    .umasks = adl_grt_machine_clears,
  },
  { .name   = "MEM_BOUND_STALLS",
    .desc   = "Counts the number of cycles the core is stalled due to a demand load which hit in the L2 cache",
    .code   = 0x0034,
    .modmsk = INTEL_V2_ATTRS,
    .cntmsk = 0x3full,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(adl_grt_mem_bound_stalls),
    .umasks = adl_grt_mem_bound_stalls,
  },
  { .name   = "MEM_LOAD_UOPS_RETIRED",
    .desc   = "Counts the number of load uops retired that hit in the L2 cache",
    .code   = 0x00d1,
    .modmsk = INTEL_V2_ATTRS,
    .cntmsk = 0x3full,
    .ngrp   = 1,
    .flags  = INTEL_X86_PEBS,
    .numasks= LIBPFM_ARRAY_SIZE(adl_grt_mem_load_uops_retired),
    .umasks = adl_grt_mem_load_uops_retired,
  },
  { .name   = "MEM_SCHEDULER_BLOCK",
    .desc   = "Counts the number of cycles that uops are blocked due to a store buffer full condition",
    .code   = 0x0004,
    .modmsk = INTEL_V2_ATTRS,
    .cntmsk = 0x3full,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(adl_grt_mem_scheduler_block),
    .umasks = adl_grt_mem_scheduler_block,
  },
  { .name   = "MEM_UOPS_RETIRED",
    .desc   = "Counts the number of tagged loads with an instruction latency that exceeds or equals the threshold of 4 cycles as defined in MEC_CR_PEBS_LD_LAT_THRESHOLD (3F6H). Only counts with PEBS enabled",
    .code   = 0x00d0,
    .modmsk = INTEL_V2_ATTRS | _INTEL_X86_ATTR_LDLAT,
    .cntmsk = 0x3ull,
    .ngrp   = 1,
    .flags  = INTEL_X86_PEBS,
    .numasks= LIBPFM_ARRAY_SIZE(adl_grt_mem_uops_retired),
    .umasks = adl_grt_mem_uops_retired,
  },
  { .name   = "MISC_RETIRED",
    .desc   = "Counts the number of LBR entries recorded. Requires LBRs to be enabled in IA32_LBR_CTL. [This event is alias to LBR_INSERTS.ANY]",
    .code   = 0x00e4,
    .modmsk = INTEL_V2_ATTRS,
    .cntmsk = 0x3full,
    .ngrp   = 1,
    .flags  = INTEL_X86_PEBS,
    .numasks= LIBPFM_ARRAY_SIZE(adl_grt_misc_retired),
    .umasks = adl_grt_misc_retired,
  },
  { .name   = "OCR0",
    .desc   = "Counts demand data reads that have any type of response",
    .code   = 0x01b7,
    .modmsk = INTEL_V2_ATTRS,
    .cntmsk = 0x3full,
    .ngrp   = 1,
    .flags  = INTEL_X86_NHM_OFFCORE,
    .numasks= LIBPFM_ARRAY_SIZE(adl_grt_ocr),
    .umasks = adl_grt_ocr,
  },
  { .name   = "OCR1",
    .desc   = "Counts demand data reads that have any type of response",
    .code   = 0x02b7,
    .modmsk = INTEL_V2_ATTRS,
    .cntmsk = 0x3full,
    .ngrp   = 1,
    .flags  = INTEL_X86_NHM_OFFCORE,
    .numasks= LIBPFM_ARRAY_SIZE(adl_grt_ocr),
    .umasks = adl_grt_ocr,
  },
  { .name   = "SERIALIZATION",
    .desc   = "Counts the number of issue slots not consumed by the backend due to a micro-sequencer (MS) scoreboard, which stalls the front-end from issuing from the UROM until a specified older uop retires",
    .code   = 0x0075,
    .modmsk = INTEL_V2_ATTRS,
    .cntmsk = 0x3full,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(adl_grt_serialization),
    .umasks = adl_grt_serialization,
  },
  { .name   = "TOPDOWN_BAD_SPECULATION",
    .desc   = "Counts the total number of issue slots that were not consumed by the backend because allocation is stalled due to a mispredicted jump or a machine clear",
    .code   = 0x0073,
    .modmsk = INTEL_V2_ATTRS,
    .cntmsk = 0x3full,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(adl_grt_topdown_bad_speculation),
    .umasks = adl_grt_topdown_bad_speculation,
  },
  { .name   = "TOPDOWN_BE_BOUND",
    .desc   = "Counts the total number of issue slots every cycle that were not consumed by the backend due to backend stalls",
    .code   = 0x0074,
    .modmsk = INTEL_V2_ATTRS,
    .cntmsk = 0x3full,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(adl_grt_topdown_be_bound),
    .umasks = adl_grt_topdown_be_bound,
  },
  { .name   = "TOPDOWN_FE_BOUND",
    .desc   = "Counts the total number of issue slots every cycle that were not consumed by the backend due to frontend stalls",
    .code   = 0x0071,
    .modmsk = INTEL_V2_ATTRS,
    .cntmsk = 0x3full,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(adl_grt_topdown_fe_bound),
    .umasks = adl_grt_topdown_fe_bound,
  },
  { .name   = "TOPDOWN_RETIRING",
    .desc   = "Counts the total number of consumed retirement slots",
    .code   = 0x00c2,
    .modmsk = INTEL_V2_ATTRS,
    .cntmsk = 0x3full,
    .ngrp   = 1,
    .flags  = INTEL_X86_PEBS | INTEL_X86_CODE_DUP,
    .numasks= LIBPFM_ARRAY_SIZE(adl_grt_topdown_retiring),
    .umasks = adl_grt_topdown_retiring,
  },
  { .name   = "UOPS_RETIRED",
    .desc   = "Counts the total number of uops retired",
    .code   = 0x00c2,
    .modmsk = INTEL_V2_ATTRS,
    .cntmsk = 0x3full,
    .ngrp   = 1,
    .flags  = INTEL_X86_PEBS | INTEL_X86_CODE_DUP,
    .numasks= LIBPFM_ARRAY_SIZE(adl_grt_uops_retired),
    .umasks = adl_grt_uops_retired,
  },
};
/* 26 events available */
